//=============================================================================
// TRP_SkitMZ_ExEmoteAnimation.js
//=============================================================================
// このソフトウェアの一部にMITライセンス、ならびにBSDライセンスで配布されている製作物が含まれています。
// http://www.opensource.org/licenses/mit-license
// https://raw.github.com/danro/easing-js/master/LICENSE

//=============================================================================
/*:
 * @target MZ
 * @author Thirop
 * @orderAfter TRP_SkitMZ
 * @plugindesc 漫符アニメプラグイン
 *
 * @help
 * ※TRP_SkitMZと併用時はバージョンが1.03以上である必要があります。
 * （TRP_SkitMZが無くても動作します。）
 *
 * 「アニメーションID始め」から「アニメーションID終わり」までのアニメーションIDを
 * 独自仕様のアニメーションの再生に使用します。
 * ツクールのデータベース上でも「最大数の変更」でアニメーションID終わり
 * までデータが存在するように調整してください。
 *
 * 通常のアニメーションの再生と同じ要領で独自再生のアニメーションを再生できます。
 * 制御文字で立ち絵にアニメーションを再生する場合は\SA[501]など
 *
 * アニメーションを編集・作成する場合は、再生するコマンドの直前にプラグインコマンドで
 * 「編集モード予約」を実行することでエディタを呼び出せます。
 * （MVコマンドは「editEmote」）
 *
 *
 * 詳しい導入方法は同梱のreadme.txtをご参照！
 *
 *
 *
 * 【更新履歴】
 * 1.03 2022/10/28 MV形式コマンド対応など
 * 1.02 2020/12/23 SE/フラッシュ再生対応など
 * 1.01 2020/11/28 エディタ改善。フレームアニメ追加など
 * 1.00 2020/11/15 初版
 *
 *
 *
 *
 *
 * @command edit
 * @text 編集モード予約
 * @desc 直後のアニメーション再生時にエディタを起動
 *
 *
 *
 * ================================================================
 * 
 * @param animationId
 * @text アニメーションID始め
 * @desc 指定したアニメーションIDから連番でエモートアニメーション用に使用します。
 * @type number
 * @default 501
 *
 * @param animationIdEnd
 * @text アニメーションID終わり
 * @desc 指定したアニメーションIDまでをエモートアニメーション用に使用します。
 * @type number
 * @default 600
 *
 *
 * @param categoryEditor
 * @text 【エディタ設定】
 * @default ==============================
 *
 * @param defaultImage
 * @text デフォルト画像名
 * @desc デフォルトのアニメーション画像名。animationsフォルダから選択。
 * @type file
 * @default img/animations/Emote
 * @parent categoryEditor
 *
 * @param reverseShift
 * @text Shiftキー機能反転
 * @desc OFFではShiftを押しながら矢印キーで値の変更幅が大きくなり、ONでは逆にShiftキーなしで大きくなります。
 * @type boolean
 * @default false
 * @parent categoryEditor
 *
 * @param enableWriteAnimation
 * @text Animation.json上書き有効化
 * @desc ONにするとエディタ上で名前の変更が可能。Animations.jsonをバックアップして各自の責任でご利用を。
 * @type boolean
 * @default false
 * @parent categoryEditor
 *
 */
//============================================================================= 



var TRP_CORE = TRP_CORE || {};

var $dataTrpAnimation = null;

(function(){
'use strict';

var pluginName = 'TRP_SkitMZ_ExEmoteAnimation';
var parameters = PluginManager.parameters(pluginName)
parameters.animationId = Number(parameters.animationId);
parameters.animationIdEnd = Number(parameters.animationIdEnd);
parameters.reverseShift = parameters.reverseShift==='true';
parameters.enableWriteAnimation = parameters.enableWriteAnimation==='true';

var isMZ = Utils.RPGMAKER_NAME === "MZ";


if(isMZ){
    PluginManager.registerCommand(pluginName, 'edit', function(args){
        if(!Utils.isOptionValid('test')){
            return;
        }
        Editor.reserveEdit = true;
    });
}

var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command,args){
    if(command === 'editEmote'){ 
        if(!Utils.isOptionValid('test')){
            return;
        }
        Editor.reserveEdit = true;
    }else{
        _Game_Interpreter_pluginCommand.call(this,...arguments);
    }
};





var _TRP_CORE_isAnimationExists = TRP_CORE.isAnimationExists;
TRP_CORE.isAnimationExists = function(animationId){
    if(this.isTrpAnimation(animationId)){
        return true;
    }else{
        return _TRP_CORE_isAnimationExists.call(this,animationId);
    }
};

TRP_CORE.isTrpAnimation = function(animationId){
    if(animationId>=parameters.animationId && animationId<=parameters.animationIdEnd){
        var trpAnimId = animationId-parameters.animationId;
        if(Utils.isOptionValid('test') && Editor.reserveEdit && !$dataTrpAnimation[trpAnimId]){
            $dataTrpAnimation[trpAnimId] = Editor.createNewData(trpAnimId);
        }
        return true;
    }else{
        return false;
    }
};



//=============================================================================
// ImageManager
//=============================================================================
ImageManager.loadSkitEmote = function(filename) {
    return this.loadBitmap("img/emotions/", filename);
};

//=============================================================================
// DataManager
//=============================================================================
var DATA_FILE_PATH = 'dataEx/TrpAnimation.json';
DataManager._databaseFiles.push({
    name:'$dataTrpAnimation',
    src:'../'+DATA_FILE_PATH
})


var _DataManager_loadDataFile = DataManager.loadDataFile;
DataManager.loadDataFile = function(name, src) {
    if(name==="$dataTrpAnimation"){
        src = src.replace('Test_','');
    }

    _DataManager_loadDataFile.call(this,name,src);
};



//=============================================================================
// Game_Temp
//=============================================================================
if(isMZ)(()=>{
    var _Game_Temp_requestAnimation = Game_Temp.prototype.requestAnimation;
    Game_Temp.prototype.requestAnimation = function(
        targets, animationId, mirror = false
    ) {
        if(TRP_CORE.isTrpAnimation(animationId)){
            this._requestSkitEmoteAnimation(targets,animationId,mirror);
        }else{
            _Game_Temp_requestAnimation.apply(this,arguments);
        }
    };

    Game_Temp.prototype._requestSkitEmoteAnimation = function(targets,animationId,mirror){
        const request = {
            targets: targets,
            animationId: animationId,
            mirror: mirror
        };
        this._animationQueue.push(request);
        for (const target of targets) {
            if (target.startAnimation) {
                target.startAnimation();
            }
        }
    };
})();



//=============================================================================
// Spriteset_Base
//=============================================================================
if(isMZ)(()=>{
    var _Spriteset_Base_createAnimation = Spriteset_Base.prototype.createAnimation;
    Spriteset_Base.prototype.createAnimation = function(request) {
        if(TRP_CORE.isTrpAnimation(request.animationId)){
            this._createSkitEmoteAnimation(request);
        }else{
            _Spriteset_Base_createAnimation.apply(this,arguments);
        }
    };

    Spriteset_Base.prototype._createSkitEmoteAnimation = function(request){
        const animation = $dataTrpAnimation[request.animationId-parameters.animationId];
        const targets = request.targets;
        const mirror = request.mirror;
        let delay = this.animationBaseDelay();
        const nextDelay = this.animationNextDelay();
        if (this.isAnimationForEach(animation)) {
            for (const target of targets) {
                this._createSkitEmoteAnimationSprite([target], animation, mirror, delay);
                delay += nextDelay;
            }
        } else {
            this._createSkitEmoteAnimationSprite(targets, animation, mirror, delay);
        }
    };

    Spriteset_Base.prototype._createSkitEmoteAnimationSprite = function(
        targets, animation, mirror, delay
    ) {
        const sprite = new Sprite_AnimationSkitEmote();
        const targetSprites = this.makeTargetSprites(targets);
        const baseDelay = this.animationBaseDelay();
        const previous = delay > baseDelay ? this.lastAnimationSprite() : null;
        if (this.animationShouldMirror(targets[0])) {
            mirror = !mirror;
        }
        sprite.targetObjects = targets;
        sprite.setup(targetSprites, animation, mirror, delay, previous);

        if((targetSprites[0] instanceof Sprite_Picture) && targetSprites[0]._isBustPicture){
            this._bustPictureEffectsContainer.addChild(sprite);
        }else{
            this._effectsContainer.addChild(sprite);
        }
        this._animationSprites.push(sprite);
    };
})();




//=============================================================================
// Sprite_Base
//=============================================================================
if(!isMZ)(()=>{
    var _Sprite_Base_startAnimation = Sprite_Base.prototype.startAnimation;
    Sprite_Base.prototype.startAnimation = function(animation, mirror, delay) {       
        if(TRP_CORE.isTrpAnimation(animation.id)){
            this._startTrpEmoteAnimation(animation,mirror,delay);
        }else{
            _Sprite_Base_startAnimation.call(this,...arguments);
        }
    };
    Sprite_Base.prototype._startTrpEmoteAnimation = function(animation,mirror,delay){
        var emoteAnimation = $dataTrpAnimation[animation.id-parameters.animationId];
        var sprite = new Sprite_AnimationSkitEmote();
        sprite.setup(this._effectTarget, emoteAnimation, mirror, delay);
        this.parent.addChild(sprite);
        this._animationSprites.push(sprite);
    };
})();




//=============================================================================
// Sprite_AnimationSkitEmote
//=============================================================================
function Sprite_AnimationSkitEmote() {
    this.initialize(...arguments);
}

var Sprite_AnimationMV = window.Sprite_AnimationMV || window.Sprite_Animation;

Sprite_AnimationSkitEmote.prototype = Object.create(Sprite.prototype);
Sprite_AnimationSkitEmote.prototype.constructor = Sprite_AnimationSkitEmote;

Sprite_AnimationSkitEmote.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.initMembers();
    this.anchor.set(0.5,0.5);
};

Sprite_AnimationSkitEmote.prototype.initMembers = function() {
    this.clear();
};

Sprite_AnimationSkitEmote.prototype.clear = function(){
    if(isMZ){
        this._targets = [];
    }else{
        this._target = null;
    }
    this._animation = null;
    this._mirror = false;
    this._delay = 0;
    this._isReady = false;
    this._rate = 1;

    this._cellSprites = [];
    this._screenFlashSprite = null;

    this.clearTiming();

    this.z = 10;
};
Sprite_AnimationSkitEmote.prototype.clearTiming = function(){
    this._frame = 0;
    this._flashColor = [0,0,0,0];
    this._flashDuration = 0;
    this._screenFlashDuration = 0;
    this._hidingDuration = 0;
};

// prettier-ignore
Sprite_AnimationSkitEmote.prototype.setup = function(
    targets, animation, mirror, delay
) {
    this.clear();

    if(isMZ){
        this._targets = targets;
    }else{
        this._target = targets;
    }
    this._animation = animation;
    this._mirror = mirror;
    this._delay = delay;

    if(mirror){
        this.scale.x = -1;
    }
    this.createScreenFlashSprite();

    this.setupCells();
    this.updateMain();
};


Sprite_AnimationSkitEmote.prototype.setupCells = function(){
    var sprites = this._cellSprites;
    var cellData = this._animation.cells;
    var length = cellData.length;
    var i,data,sprite;
    for(i = 0; i<length; i=(i+1)|0){
        data = cellData[i];
        sprite = sprites[i];
        if(sprite){
            sprite.visible = true;
        }else{
            sprite = new Sprite_AnimationSkitEmoteCell();
            this.addChild(sprite);
            sprites[i] = sprite;
        }
        sprite.setup(data);
    }

    length = sprites.length;
    for(;i<length;i=(i+1)|0){
        sprite = sprites[i];
        sprite.visible = false;
    }
};

Sprite_AnimationSkitEmote.prototype.absoluteX = function() {
    return Sprite_AnimationMV.prototype.absoluteX.call(this);
};
Sprite_AnimationSkitEmote.prototype.absoluteY = function() {
    return Sprite_AnimationMV.prototype.absoluteY.call(this);
};
Sprite_AnimationSkitEmote.prototype.isReady = function() {
    for(const cell of this._cellSprites){
        if(!cell.isReady())return false;
    }
    return true;
};
Sprite_AnimationSkitEmote.prototype.isPlaying = function() {
    for(const cell of this._cellSprites){
        if(cell.isPlaying())return true;
    }
    return false;
};

Sprite_AnimationSkitEmote.prototype.update = Sprite_AnimationMV.prototype.update;
Sprite_AnimationSkitEmote.prototype.updateMain = function(noFrameExceed=false,noExecuteTiming=false){
    if(!this._isReady){
        if(this.isReady()){
            this._isReady = true;
        }else{
            return;
        }
    }

    if (this._delay > 0) {
        this._delay--;
        return;
    }

    if(this.isPlaying()){
        this.tryProcessTimingData(this._frame);
        this._frame += 1;

        for(const child of this._cellSprites){
            child.updateMain(noFrameExceed);   
        }
        this.updatePosition();


        if (!this.isPlaying()){
            this.onEnd();
        }
    }
};

Sprite_AnimationSkitEmote.prototype.updatePosition = function() {
    Sprite_AnimationMV.prototype.updatePosition.call(this);
};

Sprite_AnimationSkitEmote.prototype.tryProcessTimingData = function(frame){
    var timings = this._animation.timings;
    if(!timings)return;

    var length = timings.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var timing = timings[i];
        if(timing.frame === frame){
            this.processTimingData(timing);
        }
    }
};



Sprite_AnimationSkitEmote.prototype.processTimingData = Sprite_AnimationMV.prototype.processTimingData;

if(isMZ){
    Sprite_AnimationSkitEmote.prototype.onEnd = Sprite_AnimationMV.prototype.onEnd;
}else{
    Sprite_AnimationSkitEmote.prototype.onEnd = function(){
        this._flashDuration = 0;
        this._screenFlashDuration = 0;
        this._hidingDuration = 0;
        if(this._target){
            this._target.setBlendColor([0,0,0,0]);
            this._target.show();
        }
    };
    Sprite_AnimationSkitEmote.prototype.remove = Sprite_AnimationMV.prototype.remove;
}


Sprite_AnimationSkitEmote.prototype.startFlash = Sprite_AnimationMV.prototype.startFlash;
Sprite_AnimationSkitEmote.prototype.updateFlash = Sprite_AnimationMV.prototype.updateFlash;

Sprite_AnimationSkitEmote.prototype.startScreenFlash = Sprite_AnimationMV.prototype.startScreenFlash;
Sprite_AnimationSkitEmote.prototype.createScreenFlashSprite = Sprite_AnimationMV.prototype.createScreenFlashSprite;
Sprite_AnimationSkitEmote.prototype.updateScreenFlash = Sprite_AnimationMV.prototype.updateScreenFlash;
Sprite_AnimationSkitEmote.prototype.absoluteX = Sprite_AnimationMV.prototype.absoluteX;
Sprite_AnimationSkitEmote.prototype.absoluteY = Sprite_AnimationMV.prototype.absoluteY;

Sprite_AnimationSkitEmote.prototype.startHiding = Sprite_AnimationMV.prototype.startHiding;
Sprite_AnimationSkitEmote.prototype.updateHiding = Sprite_AnimationMV.prototype.updateHiding;

//=============================================================================
// Sprite_AnimationSkitEmoteCell
//=============================================================================
function Sprite_AnimationSkitEmoteCell(){
    this.initialize.apply(this, arguments);
};

Sprite_AnimationSkitEmoteCell.DATA_INDEXES = {
    x:0,
    y:1,
    scaleX:2,
    scaleY:3,
    angle:4,
    opacity:5,
    animation:6
};
var IDXES = Sprite_AnimationSkitEmoteCell.DATA_INDEXES;

Sprite_AnimationSkitEmoteCell.prototype = Object.create(Sprite.prototype);
Sprite_AnimationSkitEmoteCell.prototype.constructor = Sprite_AnimationSkitEmoteCell;
Sprite_AnimationSkitEmoteCell.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);
    this.initMembers();
    this.anchor.set(0.5,0.5);
};

Sprite_AnimationSkitEmoteCell.prototype.initMembers = function(){
    this.clear();
};
Sprite_AnimationSkitEmoteCell.prototype.clear = function(){
    this.x = 0;
    this.y = 0;
    this.scale.set(1,1);
    this.opacity = 255;
    this.rotation = 0;

    this._data = null;
    this._scale = 1;
    this._rotation = 0;
    this._mirror = false;

    this._frameIndex = 0;
    this._image = '';
    this._imageChanger = null;
    this._imageChangeSpeed = 4;
    this._imageChangeCount = 0;
    this._imageMirror = false;
    this._size = 192;

    this.clearMotionParams();
};
Sprite_AnimationSkitEmoteCell.prototype.clearMotionParams = function(){
    this._frameCount = 0;

    this._dX = 0;
    this._dY = 0;
    this._dScaleX = 0;
    this._dScaleY = 0;
    this._dOpacity = 0;
    this._dRotation = 0;

    this._doPos = false;
    this._doScale = false;
    this._doOpacity = false;
    this._doRotation = false;

    this._lastPos = 0;
};


Sprite_AnimationSkitEmoteCell.prototype.setup = function(data){
    this.clear();

    this._data = data;
    this._scale = data.scale===undefined ? 1 : data.scale;
    this._rotation = data.rotation||0;
    this._mirror = data.mirror||false;

    this.scale.set(this._scale*(this._mirror?-1:1),this._scale);
    this.rotation = this._rotation;

    this._imageMirror = data.imageMirror||false;
    this.loadBitmap();

    this.blendMode = data.blend||0;
    this.setupFrame(0);
};

Sprite_AnimationSkitEmoteCell.prototype.setupFrame = function(index){
    this._frameIndex = index;
    this._frameCount = 0;

    this.clearMotionParams();

    var frame = this.currentFrame();
    if(!frame){
        return;
    }
    
    var data = frame.data;
    var x = data[IDXES.x]*(this._mirror?-1:1);
    var y = data[IDXES.y];
    if(this._rotation!==0){
        this._dX = x*Math.cos(this._rotation)-y*Math.sin(this._rotation)-this.x;
        this._dY = y*Math.cos(this._rotation)+x*Math.sin(this._rotation)-this.y;
    }else{
        this._dX = x-this.x;
        this._dY = y-this.y;
    }

    this._doPos = (this._dX || this._dY)!==0;

    this._dScaleX = data[IDXES.scaleX]*this._scale*(this._mirror?-1:1)*(this._imageMirror?-1:1)-this.scale.x;
    this._dScaleY = data[IDXES.scaleY]*this._scale-this.scale.y;
    this._doScale = (this._dScaleX||this._dScaleY)!==0;

    this._dOpacity = data[IDXES.opacity]-this.opacity;
    this._doOpacity = this._dOpacity!==0;

    this._dRotation = data[IDXES.angle]*Math.PI/180*(this._mirror?-1:1)+this._rotation-this.rotation;
    this._doRotation = this._dRotation!==0;

    var imageChanger = data[IDXES.animation]||null;
    if(imageChanger&&imageChanger.images&&imageChanger.images.length>0){
        this.setupImageChanger(imageChanger,frame.duration);
    }

    if(frame.duration===0){
        this.applyCurrentMotion();
        this.setupFrame(index+1);
    }else if(this._frameIndex===0){
        this.applyCurrentMotion();
        this.clearMotionParams();
    }
};
Sprite_AnimationSkitEmoteCell.prototype.applyCurrentMotion = function(){
    if(this._doPos){
        this.x += this._dX;
        this.y += this._dY;
    }
    if(this._doScale){
        this.scale.set(this.scale.x+this._dScaleX,this.scale.y+this._dScaleY);
    }
    if(this._doOpacity){
        this.opacity += this._dOpacity;
    }
    if(this._doRotation){
        this.rotation += this._dRotation;
    }

    // if(this._imageChanger){
    //     if(!this._imageChanger.loop){
    //         this._changeImage(this._imageChanger.images[this._imageChanger.images.length-1]);
    //     }
    // }
};

Sprite_AnimationSkitEmoteCell.prototype._changeImage = function(image){
    this._image = image;
    if(image.indexOf('ANIM:')===0){
        var elems = image.substring(5).split(':');
        var animName = elems[0];
        var index = elems[1];

        var sizeMatch = animName.match(/_x([0-9]+)$/);
        if(sizeMatch){
            this._size = Number(sizeMatch[1]);
        }else{
            this._size = 192;
        }

        ImageManager.loadAnimation(animName).addLoadListener(function(image,index,bitmap){
            if(this._image === image){
                this.bitmap = bitmap;
                var col = index%5;
                var row = Math.floor(index/5);
                var size = this._size;
                this.setFrame(col*size,row*size,size,size);
            }
        }.bind(this,image,index));
    }else{
        ImageManager.loadSkitEmote(this._image).addLoadListener(function(image,bitmap){
            if(this._image === image){
                this.bitmap = bitmap;
            }
        }.bind(this,image))
    }
};
Sprite_AnimationSkitEmoteCell.prototype.setupImageChanger = function(imageChanger,duration){
    this._imageChanger = imageChanger;
    this._imageChangeCount = 0;

    if(this._imageMirror !== imageChanger.mirror){
        this._imageMirror = imageChanger.mirror;
        this.scale.x *= -1;
    }

    if(imageChanger.speed<=0){
        this._imageChangeSpeed = Math.floor(duration/imageChanger.images.length)||1;
    }else{
        this._imageChangeSpeed = imageChanger.speed;
    }

    this._changeImage(imageChanger.images[0])
};
Sprite_AnimationSkitEmoteCell.prototype.updateImageChanger = function(){
    this._imageChangeCount += 1;
    if(this._imageChangeCount%this._imageChangeSpeed!==0){
        return;
    }

    var idx = this._imageChangeCount/this._imageChangeSpeed;
    if(idx>=this._imageChanger.images.length && !this._imageChanger.loop){
        return;
    }

    idx = idx%this._imageChanger.images.length;
    this._changeImage(this._imageChanger.images[idx]);
}; 

Sprite_AnimationSkitEmoteCell.prototype.currentFrame = function(){
    return this._data.frames[this._frameIndex];
};


Sprite_AnimationSkitEmoteCell.prototype.isReady = function() {
    return this.bitmap && this.bitmap.isReady();
};

Sprite_AnimationSkitEmoteCell.prototype.isPlaying = function() {
    return this._frameIndex < this._data.frames.length;
};
Sprite_AnimationSkitEmoteCell.prototype.loadBitmap = function() {
    this._changeImage(this._data.image);
};

Sprite_AnimationSkitEmoteCell.prototype.updateMain = function(noFrameExceed=false){
    if(!this.isPlaying()){
        this.visible = false;
        return;
    }

    this._frameCount += 1;

    var frame = this.currentFrame();
    var pos = TRP_CORE.easing[frame.easing](
        this._frameCount/frame.duration
    );
    var dp = pos - this._lastPos;
    this._lastPos = pos;

    if(this._doPos){
        this.x += this._dX*dp;
        this.y += this._dY*dp;
    }
    if(this._doScale){
        this.scale.set(
            this.scale.x+this._dScaleX*dp,
            this.scale.y+this._dScaleY*dp
        );
    }
    if(this._doOpacity){
        this.opacity += this._dOpacity*dp;
    }
    if(this._doRotation){
        this.rotation += this._dRotation*dp;
    }

    if(!!this._imageChanger){
        this.updateImageChanger();
    }


    if(this._frameCount>=frame.duration && !noFrameExceed){
        this.setupFrame(this._frameIndex+1);
    }
};





//=============================================================================
// Animation Editor
//=============================================================================
if(!Utils.isOptionValid('test'))return;

if(isMZ)(()=>{
    var _Spriteset_Base__createSkitEmoteAnimationSprite = Spriteset_Base.prototype._createSkitEmoteAnimationSprite;
    Spriteset_Base.prototype._createSkitEmoteAnimationSprite = function(
        targets, animation, mirror, delay
    ) {
        _Spriteset_Base__createSkitEmoteAnimationSprite.apply(this,arguments);

        if(Editor.reserveEdit){
            var sprite = this._animationSprites[this._animationSprites.length-1];
            if(sprite && sprite instanceof Sprite_AnimationSkitEmote){
                Editor.start(sprite);
            }
            Editor.reserveEdit = false;
        }
    };

    var _Spriteset_Base_removeAnimation = Spriteset_Base.prototype.removeAnimation;
    Spriteset_Base.prototype.removeAnimation = function(sprite) {
        if(Editor.isActive){
            return;
        }
        _Spriteset_Base_removeAnimation.call(this,sprite);
    };
})();
if(!isMZ)(()=>{
    var _Sprite_Base__startTrpEmoteAnimation = Sprite_Base.prototype._startTrpEmoteAnimation;
    Sprite_Base.prototype._startTrpEmoteAnimation = function(animation,mirror,delay){
        _Sprite_Base__startTrpEmoteAnimation.call(this,...arguments);

        if(Editor.reserveEdit){
            var sprite = this._animationSprites[this._animationSprites.length-1];
            if(sprite && sprite instanceof Sprite_AnimationSkitEmote){
                Editor.start(sprite);
            }
            Editor.reserveEdit = false;
        }
    };
})();



var _Sprite_AnimationSkitEmoteCell_initMembers = Sprite_AnimationSkitEmoteCell.prototype.initMembers;
Sprite_AnimationSkitEmoteCell.prototype.initMembers = function(){
    _Sprite_AnimationSkitEmoteCell_initMembers.call(this);
    this._opacityRate = 1;
    this._opacity = this.opacity;
    this._muteTiming = false;
};

var _Sprite_AnimationSkitEmoteCell_setup = Sprite_AnimationSkitEmoteCell.prototype.setup;
Sprite_AnimationSkitEmoteCell.prototype.setup = function(data){
    _Sprite_AnimationSkitEmoteCell_setup.call(this,data);
    this._opacity = this.opacity;
};
var _Sprite_AnimationSkitEmoteCell_updateMain = Sprite_AnimationSkitEmoteCell.prototype.updateMain;
Sprite_AnimationSkitEmoteCell.prototype.updateMain = function(){
    this.opacity = this._opacity;
    _Sprite_AnimationSkitEmoteCell_updateMain.call(this);
    this._opacity = this.opacity;

    this.opacity *= this._opacityRate;
};

var _Sprite_AnimationSkitEmote_tryProcessTimingData = Sprite_AnimationSkitEmote.prototype.tryProcessTimingData;
Sprite_AnimationSkitEmote.prototype.tryProcessTimingData = function(frame){
    if(this._muteTiming)return;

    _Sprite_AnimationSkitEmote_tryProcessTimingData.call(this,frame);
};


//=============================================================================
// Editor
//=============================================================================
var Editor = TRP_CORE.EmoteAnimationEditor = function Editor(){
    this.initialize.apply(this, arguments);
};



var backspaceKey = Input.keyMapper[8]||'backspace';
Input.keyMapper[8] = backspaceKey;
var animationKey = Input.keyMapper[65]||'a';
Input.keyMapper[65] = animationKey;
var blendKey = Input.keyMapper[66]||'b';
Input.keyMapper[66] = blendKey;
var cellKey = Input.keyMapper[67]||'c';
Input.keyMapper[67] = cellKey;
var durationKey = Input.keyMapper[68]||'d';
Input.keyMapper[68] = durationKey;
var easingKey = Input.keyMapper[69]||'e';
Input.keyMapper[69] = easingKey;
var frameKey = Input.keyMapper[70]||'f';
Input.keyMapper[70] = frameKey;
var guideKey = Input.keyMapper[71]||'g';
Input.keyMapper[71] = guideKey;
var imageKey = Input.keyMapper[73]||'i';
Input.keyMapper[73] = imageKey;
var lockKey = Input.keyMapper[76]||'l';
Input.keyMapper[76] = lockKey;
var motionKey = Input.keyMapper[77]||'m';
Input.keyMapper[77] = motionKey;
var nameKey = Input.keyMapper[78]||'n';
Input.keyMapper[78] = nameKey;
var opacityKey = Input.keyMapper[79]||'o';
Input.keyMapper[79] = opacityKey;
var posKey = Input.keyMapper[80]||'p';
Input.keyMapper[80] = posKey;
var replayKey = Input.keyMapper[82]||'r';
Input.keyMapper[82] = replayKey;
var saveKey = Input.keyMapper[83]||'s';
Input.keyMapper[83] = saveKey;
var timingKey = Input.keyMapper[84]||'t';
Input.keyMapper[84] = timingKey;
var viewKey = Input.keyMapper[86]||'v';
Input.keyMapper[86] = viewKey;


var KEY_CODE = {
    backSpace : 8,
    num : 48,
    alphabet : 65,
    alphabetEnd: 90,
    tenkey:96,
    minus:189,
    tenkeyMinus:109,
    dot:190,
    tenkeyDot:110,
};


Editor.reserveEdit = false;
Editor._savedUpdate = null;
Editor.isActive = false;

Editor.start = function(sprite){
    this.isActive = true;

    this._savedUpdate = SceneManager._scene.update;
    var editor = new Editor(sprite)
    var scene = SceneManager._scene;
    scene.addChild(editor);

    scene.update = function(){
        if(!editor.update()){
            SceneManager._scene.update = this._savedUpdate;

            this._savedUpdate = null;
            scene.removeChild(editor);
            editor.destroy();
            this.isActive = false;
        }
    }.bind(this);
};



Editor.prototype = Object.create(Sprite.prototype);
Editor.prototype.constructor = Sprite_AnimationSkitEmoteCell;
Editor.prototype.initialize = function(sprite){
    Sprite.prototype.initialize.call(this);
    this.initMembers();

    this._animationSprite = sprite;
    this._animation = sprite._animation;

    SoundManager.playOk();

    this.createLines();
    this.createCenterSprite();

    this.refreshAllLines();

    this.selectCell(0);

    this.setMode('edit');

    this.startPreview();
};
Editor.prototype.initMembers = function(){
    this.clearInputtingModeParams();
    this.clearKeyDownListeneres();

    this._animationSprite = null;
    this._animation = null;
    this._cellIndex = -1;
    this._frameIndex = -1;
    this._timingIndex = -1;

    this._end = false;
    this._mode = '';
    this._timingMode = false;
    this._lastMode = '';
    this._preview = false;
    this._loopPreview = false;
    this._muteTiming = false;
    this._viewMode = 0;
    this._scaleRateFix = true;

    this._headerLine = null;
    this._cellLine = null;
    this._imageLine = null;
    this._frameLine = null;

    this._editHeaderLine = null;
    this._motionLine = null;
    this._posLine = null;
    this._scaleLine = null;
    this._angleLine = null;
    this._opacityLine = null;

    this._timingHeaderLine = null;
    this._timingLine = null;
    this._timingFrameLine = null;
    this._seHeaderLine = null;
    this._seLine = null;
    this._seSettingLine = null;
    this._flashHeaderLine = null;
    this._flashLine = null;
    this._flashSettingLine = null;

    this._activePicker = null;
    this._easingPicker = null;
    this._imagePicker = null;
    this._animationPicker = null;
    this._sePicker = null;
    this._colorPicker = null;

    this._activeWindow = null;

    this._nameInput = null;
    this._inputWindow = null;

    this._centerSprite = null;

    this._tempGuideSprites = [];
    this._guideSprite = new GuideSprite();
    this.addChild(this._guideSprite);
};
Editor.prototype.animationDBId = function(){
    return this._animation.id+parameters.animationId-1;
};
Editor.prototype.animationDBData = function(){
    return $dataAnimations[this.animationDBId()];
};
Editor.prototype.currentCell = function(){
    return this._animation.cells[this._cellIndex];
};
Editor.prototype.currentFrame = function(){
    return this.currentCell().frames[this._frameIndex];
};
Editor.prototype.timingData = function(){
    if(!this._animation.timings){
        this._animation.timings = [];
    }
    var timings = this._animation.timings;
    if(timings.length===0){
        this.addTiming(0,timings);
    }
    return timings;
};
Editor.prototype.currentTiming = function(){
    return this.timingData()[this._timingIndex];
}

Editor.prototype.restoreMode = function(){
    // this.setMode(this._lastMode);
    if(this._timingMode){
        this.setMode('timing');
    }else{
        this.setMode('edit');
    }
};
Editor.prototype.setMode = function(mode){
    if(this._mode === mode)return;
    this._lastMode = this._mode;

    if(this._loopPreview){
        this.endLoopPreview();
    }

    if(this._mode === 'input'){
        this.resignKeyListeners();
        this._inputLine.unfocus();
        this.clearInputtingModeParams();
    }else if(this._timingMode){
        this._endTimingMode(this._mode);
    }else{
        this._endMode(this._mode);
    }
    this._mode = mode;
    this._centerSprite.visible = false;

    if(mode === 'input'){
        this.focusLines([this._inputLine]);
        this._inputLine.focusSymbol(this._inputSymbol);
        this.registerKeyListeners();
        this.startLoopPreview();
    }else if(this._timingMode){
        this._setTimingMode(mode);
    }else{
        this._setMode(mode);
    }
};

Editor.prototype._setMode = function(mode){
    switch(mode){
    case 'editName':
        this.startEditName();
        this.focusLines([this._headerLine]);
        this._headerLine.focus(1);
        break;
    case 'editImage':
        this.startLoopPreview();
        this.focusLines([this._imageLine]);
        this._imageLine.focus(0);
        this.startPickingImage();
        break;
    case 'editAnimation':
        this.focusLines([this._animationLine]);
        this._animationLine.focus(0);
        this.startPickingAnimation();
        break;
    case 'shift':
        this.startLoopPreview();
        this.focusLines([this._cellLine]);
        this.showGuide(Editor.GUIDE.shift,'rgba(0,150,0,0.5)');
        break;
    case 'edit':
        this.focusLines(this.allLines());
        this._centerSprite.visible = true;
        this.showGuide(Editor.GUIDE.edit,'rgba(150,0,0,0.5)');
        break;
    case 'editEasing':
        this.startLoopPreview();
        this.focusLines([this._editHeaderLine,this._motionLine]);
        this._motionLine.focus(1);
        this.startPickingEasing();
        break;
    }
};
Editor.prototype._endMode = function(mode){
    switch(mode){
    case 'editName':
        this._headerLine.unfocus();
        this.endNameInput();
        break;
    case 'edit':
        break;
    case 'cell':
        this._cellLine.deactivate();
        break;
    case 'frame':
        this._frameLine.deactivate();
        break;
    case 'editImage':
        break;
    case 'editEasing':
        this._motionLine.unfocus();
        break;
    case 'editAnimation':
        this._animationLine.unfocus();
        break;
    }
};

Editor.prototype._setTimingMode = function(mode){
    switch(mode){
    case 'editFlash':
        this.focusLines([this._flashLine]);
        this._flashLine.focus(0);
        this.startPickingFlash();
        this.startLoopPreview();
        break;
    case 'timing':
        this.startLoopPreview();
        this.focusLines(this.allLines());
        this.showGuide(Editor.GUIDE.timingMode,'rgba(0,0,150,0.5)');
        break;
    case 'editSe':
        this.startLoopPreview();
        this.focusLines([this._seLine]);
        this._motionLine.focus(1);
        this.startPickingSe();
        break;
    }
};
Editor.prototype._endTimingMode = function(mode){
    switch(mode){
    case 'editFlash':
        this._flashLine.unfocus();
        break;
    case 'timing':
        break;
    case 'editSe':
        this._seLine.unfocus();
        break;
    }
};



Editor.prototype.update = function(){
    if(this._preview){
        if(!this.updatePreview()){
            return true;
        }
    }

    if(this._centerSprite.visible){
        this.updateCenterSprite();
    }

    if(this._activeWindow){
        this._activeWindow.update();
        if(this._activeWindow.isClosed()){
            this.removeChild(this._activeWindow);
            this._activeWindow.destroy();
            this._activeWindow = null;
        }
        return true;
    }
    if(this._activePicker){
        if(this._loopPreview && Input.isTriggered(replayKey) && Input.isPressed('control')){
            SoundManager.playCursor();
            this.switchLoopPreview();
        }
        this._activePicker.update();
        return true;
    }
    if(this._inputWindow){
        this._inputWindow.update();
        if(this._inputWindow.isClosed()){
            this.restoreMode();
        }
        return true;
    }

    if(this._guideSprite){
        this._guideSprite.update();
    }
    if(this._tempGuideSprites.length>0){
        this.updateTempGuide();
    }

    if(this._timingMode){
        this.updateTimingMode();
    }else{
        this.updateEditMode();
    }
    return !this._end;
};
Editor.prototype.updateEditMode = function(){
    if(TouchInput.isTriggered()){
        this.processTouch();
    }
    if(Input._latestButton){
        if(this._loopPreview && Input.isTriggered(replayKey) && Input.isPressed('control')){
            SoundManager.playCursor();
            this.switchLoopPreview();
        }else if(this._mode === 'shift'){
            this.updateInputShiftMode();
        }else{
            this.updateInput();
        }
    }
};
Editor.prototype.updateTimingMode = function(){
    if(TouchInput.isTriggered()){
        this.processTouchTimingMode();
    }
    if(Input._latestButton){
        if(this._loopPreview && Input.isTriggered(replayKey) && Input.isPressed('control')){
            SoundManager.playCursor();
            this.switchLoopPreview();
        }else{
            this.updateInputTimingMode();
        }
    }
};

Editor.prototype.processTouch = function(){
    if(this._headerLine.processTouch())return;
    if(this._cellLine.processTouch())return;
    if(this._imageLine.processTouch())return;
    if(this._frameLine.processTouch())return;
    if(this._motionLine.processTouch())return;
    if(this._scaleLine.processTouch())return;
    if(this._animationLine.processTouch())return;
};
Editor.prototype.processTouchTimingMode = function(){
    if(this._timingLine.processTouch())return;
    if(this._seLine.processTouch())return;
    if(this._seSettingLine.processTouch())return;
    if(this._flashLine.processTouch())return;
    if(this._flashSettingLine.processTouch())return;
};
Editor.prototype.updateInput = function(){
    if(!this._loopPreview && Input.isRepeated(replayKey)){
        this.startPreview();
    }else if(Input.isTriggered('cancel')){
        this.showEndWindow();
    }else if(Input.isTriggered(nameKey)){
        this.didClickNameButton();
    }else if(Input.isTriggered(posKey)){
        this.didClickPosition();
    }else if(Input.isTriggered(viewKey)){
        this.didClickViewMode();
    }else if(Input.isTriggered(imageKey)){
        this.didClickImageButton();
    }else if(Input.isTriggered(animationKey)){
        this.didClickAnimationButton();
    }else if(Input.isTriggered(blendKey)){
        this.didClickBlendButton();
    }else if(Input.isTriggered(durationKey)){
        this.didClickFrameButton();
    }else if(Input.isTriggered(easingKey)){
        this.didClickEasingButton();
    }else if(Input.isTriggered(lockKey)){
        this.didClickFixScaleRateButton();
    }else if(!Input.isPressed('control')&&Input.isTriggered(saveKey)){
        this.setMode('shift');
        SoundManager.playCursor();
    }else if(Input.isTriggered(timingKey)){
        this.startTimingMode();
    }else if(Input.isTriggered(cellKey)){
        SoundManager.playCursor();
        if(Input.isPressed('control')){
            if(Input.isPressed('shift')){
                this.addCell(this._cellIndex);
            }else{
                this.addCell(this._cellIndex+1);
            }
        }else{
            if(Input.isPressed('shift')){
                this.selectCell(this._cellIndex-1);
            }else{
                this.selectCell(this._cellIndex+1);
            }
        }
    }else if(Input.isTriggered(motionKey)){
        SoundManager.playCursor();
        if(Input.isPressed('control')){
            if(Input.isPressed('shift')){
                this.addFrame(this._frameIndex);
            }else{
                this.addFrame(this._frameIndex+1);
            }
        }else{
            if(Input.isPressed('shift')){
                this.selectFrame(this._frameIndex-1);
            }else{
                this.selectFrame(this._frameIndex+1);
            }
        }
    }else{
        if(this._mode === 'input'){
            this.updateInputInputtingMode();
        }else if(this._mode==='edit'){
            this.updateInputEditMode();
        }else if(this._mode==='editImage'){
        }else if(this._mode==='editEasing'){
        }
    }
};
Editor.prototype.updateInputTimingMode = function(){
    if(!this._loopPreview && Input.isRepeated(replayKey)){
        this.startPreview();
    }else if(Input.isTriggered('cancel')||Input.isTriggered('ok')){
        SoundManager.playCancel();
        this.endTimingMode();

    }else if(Input.isTriggered(frameKey)){
        this.didClickTimingFrameButton();
    }else if(Input.isTriggered(nameKey)){
        this.didClickSeButton();
    }else if(Input.isTriggered(viewKey)){
        this.didClickSeVolumeButton();
    }else if(Input.isTriggered(posKey)){
        this.didClickSePitchButton();
    }else if(Input.isTriggered(animationKey)){
        this.didClickSePanButton();

    }else if(Input.isTriggered(cellKey)){
        //color
        this.didClickFlashButton();
    }else if(Input.isTriggered(opacityKey)){
        this.didClickFlashOpacityButton();
    }else if(Input.isTriggered(durationKey)){
        this.didClickFlashDurationButton();
    }else if(Input.isTriggered(saveKey)){
        this.didClickFlashScopeButton();

    }else if(Input.isTriggered(timingKey)){
        SoundManager.playCursor();
        if(Input.isPressed('control')){
            if(Input.isPressed('shift')){
                this.addTiming(this._timingIndex);
            }else{
                this.addTiming(this._timingIndex+1);
            }
        }else{
            if(Input.isPressed('shift')){
                this.selectTiming(this._timingIndex-1);
            }else{
                this.selectTiming(this._timingIndex+1);
            }
        }
    }else{
        if(this._mode === 'input'){
            this.updateInputInputtingMode();
        }else{
            if(Input.isTriggered(backspaceKey)){
                this.showDeleteWindow();
            }
        }
    }
};



/* inputMode
===================================*/
Editor.prototype.clearInputtingModeParams = function(){
    this._inputSymbol = '';
    this._inputLine = null;

    this._inputting = '';
    this._inputHandler = null;
    this._inputDefValue = 0;
    this._inputMinValue = 0;
    this._inputMaxValue = 0;
};
Editor.INPUT_MODE_GUIDE = {
    footer:[
        'キーボードで数値を入力してください',
        'BackSpace：数値をクリア',
        null,
        'Esc：終了',
        'G：ガイドを隠す'
    ],

    duration:[
        '【モーションデュレーション】',
        'モーション完了までに要するフレーム数を設定',
    ],

    timing:[
        '【Se/フラッシュ発生フレーム】'
    ],

    volume:[
        '【SEボリューム】'
    ],
    pitch:[
        '【SEピッチ】'
    ],
    pan:[
        '【SEパン】'
    ],

    flashOpacity:[
        '【フラッシュ不透明度(強さ)】'
    ],
    flashDuration:[
        '【フラッシュデュレーション】',
        'フラッシュが完了までに要するフレーム数を設定',
    ],
}
Editor.prototype.startInputtingMode = function(line,symbol,handler=null,def=0,min=0,max=999999){
    this.clearInputtingModeParams();

    this._inputLine = line;
    this._inputSymbol = symbol;
    this._inputHandler = handler;

    this._inputDefValue = def;
    this._inputMinValue = min;
    this._inputMaxValue = max;

    this.setMode('input');

    var guide = Editor.INPUT_MODE_GUIDE[symbol].concat(
        Editor.INPUT_MODE_GUIDE.footer
    );
    var color = 'rgba(0,255,150,0.5)';
    this.showGuide(guide,color);
};
Editor.prototype.updateInputInputtingMode = function(){
    if(Input.isTriggered('cancel')||Input.isTriggered('ok')){
        this._inputting = String(Number(this._inputting).clamp(this._inputMinValue,this._inputMaxValue));
        this.refreshInputLine();

        this.restoreMode();
        SoundManager.playCursor();
    }
};
Editor.prototype.registerKeyListeners = function(){
    var listener = this._onKeyDown.bind(this);
    this._keydownListener = listener;
    document.addEventListener('keydown', listener);
};
Editor.prototype.clearKeyDownListeneres = function(){
    this._keydownListener = null;
};
Editor.prototype.resignKeyListeners = function(){
    if(this._keydownListener){
        document.removeEventListener('keydown',this._keydownListener);    
    }
    this.clearKeyDownListeneres();
};


Editor.prototype._onKeyDown = function(event){
    var keyCode = event.keyCode;
    var numKeyCode = KEY_CODE.num;
    var tenKeyCode = KEY_CODE.tenkey;
    var chara = '';

    if(keyCode===KEY_CODE.backSpace){
        this._inputting = '';
    }else if(keyCode>=numKeyCode&&keyCode<numKeyCode+10){
        chara = Number(keyCode-numKeyCode);
    }else if(keyCode>=tenKeyCode&&keyCode<tenKeyCode+10){
        chara = Number(keyCode-tenKeyCode);
    }else if(keyCode===KEY_CODE.minus||keyCode===KEY_CODE.tenkeyMinus){
        chara = '-';
        this._inputting = '';
    }else if(keyCode===KEY_CODE.dot||keyCode===KEY_CODE.tenkeyDot){
        if(!this._inputting.contains('.')){
            chara = '.';
        }
    }else{
        return;
    }

    SoundManager.playCursor();
    this._inputting += chara;

    var value = Number(this._inputting)||0;
    value = value.clamp(this._inputMinValue,this._inputMaxValue);
    this.refreshInputLine();


    if(this._inputHandler){
        this._inputHandler(value);
    }
};
Editor.prototype.refreshInputLine = function(){
    this._inputLine.setValue(this._inputSymbol,this._inputting);
};



/* picker
===================================*/
Editor.prototype.startPicking = function(picker){
    this._activePicker = picker;
    this.addChild(picker);
};
Editor.prototype.didEndPicking = function(){
    this._activePicker.parent.removeChild(this._activePicker);
    this._activePicker = null;


    this.restoreMode();
};


/* guide
===================================*/
Editor.prototype.showGuide = function(guide,color){
    this._guideSprite.setTexts(guide,color);
    this.addChild(this._guideSprite);
};
Editor.GUIDE = {
    shift:[
        '【セルのシフト】',
        '矢印：セルの位置を移動',
        'ctrl+矢印：セルの拡大率変更',
        'Q/W：セルを回転',
        'M：セルを反転',
        'Enter：決定',
        null,
        'ctrl+R：プレビュー切り替え',
        'Esc：終了',
        'G：ガイドを隠す'
    ],
    edit:[
        '【編集モード】',
        '各モーションの完了状態の',
        '位置、拡大率、角度、不透明度を設定。',
        'ctrlキーはMacではOptキーを使用。',
        'Shift：押しながらで調節単位を変更。',
        null,
        'C：セルの選択(+shiftで前に)',
        'M：モーションの選択(+shiftで前に)',
        'ctrl+C/M：セル/モーション挿入(+shiftで前に)',
        'S：セルの座標シフト',
        'T：SE/フラッシュ編集モード',
        null,
        'R：プレビュー',
        'Backspace：削除メニュー',
        'Esc：終了/保存メニュー',
        'G：ガイドを隠す'
    ],

    timingMode:[
        '【タイミングデータ編集】',
        'タイミングを指定してSE/フラッシュを設定。',
        null,
        'T：タイミングデータの選択(+shiftで前に)',
        'ctrl+T：タイミングデータ挿入(+shiftで前に)',
        null,
        'ctrl+R：プレビュー切り替え',
        'Backspace：削除メニュー',
        'Esc：タイミングデータ編集モード終了',
        'G：ガイドを隠す'

    ]
};
Editor.prototype.showTempGuide = function(texts){
    var guide = new TempGuideSprite(texts);
    this.addChild(guide);
    this._tempGuideSprites.push(guide);
};
Editor.prototype.updateTempGuide = function(){
    var length = this._tempGuideSprites.length;
    for(var i = length-1; i>=0; i=(i-1)|0){
        var guide = this._tempGuideSprites[i];
        guide.update();
        if(!guide.visible){
            this._tempGuideSprites.splice(i,1);
            this.removeChild(guide);
            guide.destroy();
        }
    }
};
function TempGuideSprite(){
    this.initialize.apply(this, arguments);
}
TempGuideSprite.prototype = Object.create(Sprite.prototype);
TempGuideSprite.prototype.constructor = TempGuideSprite;
TempGuideSprite.prototype.initialize = function(texts){
    var lines = texts.length;
    var lineHeight = 28;

    var width = 450;
    var height = lines*lineHeight;
    var bitmap = new Bitmap(width,height);
    Sprite.prototype.initialize.call(this,bitmap);
    this.anchor.set(0.5,0.5);
    this.x = Graphics.width/2;
    this.y = Graphics.height/2;

    this._count = 0;

    bitmap.fillAll('rgba(0,0,0,0.8)');

    bitmap.fontSize = lineHeight - 6;
    bitmap.outlineColor = 'black';
    bitmap.outlineWidth = 4;

    var y = 0;
    var length = texts.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var text = texts[i];
        bitmap.drawText(text,0,y,width,lineHeight,'center');

        y += lineHeight;
    }
};
TempGuideSprite.prototype.update = function(){
    Sprite.prototype.update.call(this);

    this._count+=1;
    if(this._count>90){
        this.opacity -= 20;
        if(this.opacity<=0){
            this.visible = false;
        }
    }
};




/* preview
===================================*/
Editor.prototype.switchLoopPreview = function(){
    if(this._preview){
        if(!this._muteTiming){
            this.setMuteTiming(true);
        }else{
            this.setMuteTiming(false);
            this.endPreview();
        }
    }else{
        this.startPreview();
    }
};
Editor.prototype.setMuteTiming = function(mute=true){
    this._muteTiming = mute;
    var cells = this._animationSprite._muteTiming = mute;
};
Editor.prototype.startLoopPreview = function(muteTiming = false){
    this._loopPreview = true;
    this._muteTiming = muteTiming;
    this.startPreview();
    this.opacity = 255;

    this.refreshViewModeOpacityRate();
};
Editor.prototype.endLoopPreview = function(){
    this._loopPreview = false;
    this.endPreview();
};

Editor.prototype.startPreview = function(){
    this._preview = true;
    this._animationSprite.visible = true;
    this._animationSprite.clearTiming();
    this._animationSprite.setupCells();
    this._animationSprite._isReady = false;

    this._centerSprite.visible = false;

    this._animationSprite.updateMain();

    this.opacity = 128;
};
Editor.prototype.endPreview = function(){
    this._preview = false;
    this._animationSprite.visible = false;
    this.opacity = 255;

    this._centerSprite.visible = true;

    this._animationSprite.onEnd();
    if(this._animationSprite._screenFlashSprite){
        this._animationSprite._screenFlashSprite.visible = false;
    }
    this.refreshViewModeOpacityRate();
    this.refreshAnimationSpriteFrame();
};
Editor.prototype.updatePreview = function(){
    this._animationSprite.update();

    if(this._loopPreview){
        if(this._animationSprite.isReady() && !this._animationSprite.isPlaying()){
            this._animationSprite.onEnd();
            this._animationSprite.clearTiming();
            this._animationSprite.setupCells();
            this._animationSprite._isReady = false;
        };
        return true;
    }else{
        if(Input.isTriggered(replayKey)){
            this.startPreview();
        }else if(
            Input.isTriggered('ok') ||
            (this._animationSprite.isReady() && !this._animationSprite.isPlaying()))
        {
            this.endPreview();
        };
        return false;
    }
};
Editor.prototype.refreshAnimationSpriteFrame = function(){
    var cell = this.currentCell();
    var frames = cell.frames;
    var count = 0;
    for(var i = 0; i<=this._frameIndex; i=(i+1)|0){
        var frame = frames[i];
        count += frame.duration;
    }

    this.freezeAnimationAtCount(count,this._cellIndex);
};

Editor.prototype.clearViewModeOpacityRate = function(){
    var cells = this._animationSprite._cellSprites;
    var length = cells.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var cell = cells[i];
        cell._opacityRate = 1;
    }
};

Editor.prototype.refreshViewModeOpacityRate = function(){
    var cells = this._animationSprite._cellSprites;
    var length = cells.length;
    var opacityRate = this.viewModeOpacity()/255;
    for(var i = 0; i<length; i=(i+1)|0){
        var cell = cells[i];
        cell._opacityRate = i===this._cellIndex ? 1 : opacityRate;
        cell.opacity = cell._opacity*cell._opacityRate;
    }
};
Editor.prototype.viewModeOpacity = function(){
    if(this._timingMode){
        return 255
    }
    return this._viewMode===0?255:(this._viewMode===1?128:0);
};
Editor.prototype.freezeAnimationAtCount = function(count,cellIndex=this._cellIndex){
    var sprite = this._animationSprite;
    sprite.setupCells();
    sprite.visible = true;

    sprite.updatePosition();
    if(count===0){
        return;
    }    

    var sprite = this._animationSprite;
    for(var i=0; i<count; i=(i+1)|0){
        sprite.updateMain(i===count-1,true);
    }
};



/* focus
===================================*/
Editor.prototype.allLines = function(){
    if(!this._timingMode){
        return [
            this._headerLine,
            this._cellLine,this._imageLine,this._frameLine,
            this._editHeaderLine,this._motionLine,this._posLine,this._scaleLine,this._angleLine,this._opacityLine,this._animationLine
        ];
    }else{
        return [
            this._timingHeaderLine,
            this._timingLine,this._timingFrameLine,
            this._seHeaderLine,this._seLine,this._seSettingLine,
            this._flashHeaderLine,this._flashLine,this._flashSettingLine
        ]
    }
};
Editor.prototype.focusLines = function(lines=[]){
    var allLines = this.allLines();
    var length = allLines.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var line = allLines[i];
        if(lines.contains(line)||line===this._headerLine){
            // line.opacity = 255;
            line.setBlendColor([0,0,0,0])
        }else{
            line.setBlendColor([-255,-255,-255,48])
        }
    }
};


/* save
===================================*/
Editor.prototype.didClickSaveButton = function(){
    var data = $dataTrpAnimation;
    var file = JSON.stringify(data,null,4);
    var fs = require('fs');
    var path = require('path');
    var base = path.dirname(process.mainModule.filename);
    var filePath = path.join(base, DATA_FILE_PATH);
    fs.writeFileSync(filePath, file);
    SoundManager.playSave();

    //save animation names
    var animationPath = 'data/Animations.json';
    file = fs.readFileSync(animationPath,{ encoding: "utf8" });

    var name = this.animationDBData().name;
    var changed = false;
    var regStr = '({"id":'+this.animationDBId()+',[^\n]*"name":")(.*?)(")'
    var regExp = new RegExp(regStr);
    file = file.replace(regExp,function(arg1,arg2,arg3,arg4){
        if(arg3!==name){
            changed = true;
            return arg2+name+arg4;
        }else{
            return arg1;
        }
    })

    if(changed){
        fs.writeFileSync(animationPath,file);
    }
};


/* cell mode
===================================*/
Editor.createNewData = function(id){
    return {
        id:id+1,
        position:0,
        cells:[
            this.createCell()
        ],
        timings:[],
    }
};
Editor.createCell = function(){
    return {
        image:this.defaultImage().replace('img/animations/',''),
        imageMirror:false,
        frames:[this.createFrame(0)],
        rotation:0,
        scale:1.0,
        mirror:false,
    }
};
Editor.createFrame = function(idx=0){
    return {
        duration:idx===0 ? 0 : 10,
        easing:'linear',
        data:[0,0,1,1,0,255]
    }
};
Editor.defaultImage = function(){
    return 'ANIM:'+parameters.defaultImage.replace('.png','')+':0';
};

Editor.prototype.addCell = function(index){
    var cells = this._animation.cells;
    if(index<0 || index>cells.length){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playOk();

    // var cell = this.createCell();
    var cell = JsonEx.makeDeepCopy(this.currentCell());
    cells.splice(index,0,cell);

    this.refreshCellLine();
    this.selectCell(index,true);
};
Editor.prototype.createCell = function(){
    return Editor.createCell();
};
Editor.prototype.createFrame = function(idx=0){
    return Editor.createFrame(idx);
};

Editor.prototype.deleteCurrentCell = function(){
    var cells = this._animation.cells;
    if(cells.length===1){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playCancel();
    cells.splice(this._cellIndex,1);

    this.refreshCellLine();

    var index = this._cellIndex.clamp(0,cells.length-1);
    this.selectCell(index,true);
};

Editor.prototype.refreshCellLine = function(){
    var line = this._cellLine;
    var cellLength = this._animation.cells.length;
    line.setSelectionElem(1,cellLength);
};

/* shift mode
===================================*/
Editor.prototype.updateInputShiftMode = function(){
    var shift = Input.isPressed('shift');
    if(parameters.reverseShift){
        shift = !shift;
    }
    if(Input.isRepeated('left')){
        if(Input.isPressed('control')){
            this.shiftScale(-1,shift);
        }else{
            this.shiftX(-1,shift);
        }
        SoundManager.playCursor();
    }else if(Input.isRepeated('right')){
        if(Input.isPressed('control')){
            this.shiftScale(1,shift);
        }else{
            this.shiftX(1,shift);
        }
        SoundManager.playCursor();
    }else if(Input.isTriggered('down')){
        if(Input.isPressed('control')){
            this.shiftScale(-1,shift);
        }else{
            this.shiftY(1,shift);
        }
        SoundManager.playCursor();
    }else if(Input.isTriggered('up')){
        if(Input.isPressed('control')){
            this.shiftScale(1,shift);
        }else{
            this.shiftY(-1,shift);
        }
        SoundManager.playCursor();
    }else if(Input.isTriggered('cancel')||Input.isTriggered('ok')){
        SoundManager.playOk();
        this.restoreMode();
    }else if(Input.isRepeated('pagedown')){
        this.shiftAngle(1,shift);
        SoundManager.playCursor();
    }else if(Input.isRepeated('pageup')){
        this.shiftAngle(-1,shift);
        SoundManager.playCursor();
    }else if(Input.isTriggered(motionKey)){
        this.mirrorCell();
        SoundManager.playCursor();
    }
};
Editor.prototype.mirrorCell = function(){
    var cell = this.currentCell();
    cell.mirror = !cell.mirror;
};
Editor.prototype.shiftX = function(sign,shift){
    var delta = sign * (shift?10:1);
    var frames = this.currentCell().frames;
    var length = frames.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var frame = frames[i];
        frame.data[IDXES.x] += delta;
    }
    this.currentAnimationCell().x += delta;
    this.refreshPosLineValues();
};
Editor.prototype.shiftY = function(sign,shift){
    var delta = sign * (shift?10:1);
    var frames = this.currentCell().frames;
    var length = frames.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var frame = frames[i];
        frame.data[IDXES.y] += delta;
    }
    this.currentAnimationCell().y += delta;
    this.refreshPosLineValues();
};
Editor.prototype.shiftAngle = function(sign,shift){
    var delta = sign * (shift?10:1) * Math.PI/180;
    var cell = this.currentCell();
    cell.rotation = (cell.rotation||0) + delta;
};
Editor.prototype.shiftScale = function(sign,shift){
    var delta = sign * (shift?0.1:0.01);
    var cell = this.currentCell();
    cell.scale = (cell.scale||1.0) + delta;
};


/* frame mode
===================================*/
Editor.prototype.addFrame = function(index){
    var cell = this.currentCell();
    var frames = cell.frames;
    if(index<0 || index>frames.length){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playOk();

    var frame = JsonEx.makeDeepCopy(this.currentFrame());
    frame.data[IDXES.animation] = null;

    if(index===0){
        frame.duration = 1;
    }else if(frame.duration<=1){
        frame.duration = 10;
    }
    frames.splice(index,0,frame);

    this.refreshFrameLine();
    this.selectFrame(index,true);
};
Editor.prototype.deleteCurrentFrame = function(){
    var cell = this.currentCell();
    var frames = cell.frames;
    if(frames.length===1){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playCancel();
    frames.splice(this._frameIndex,1);

    this.refreshFrameLine();

    var index = this._frameIndex.clamp(0,frames.length-1);
    this.selectFrame(index,true);
};
Editor.prototype.refreshFrameLine = function(){
    var line = this._frameLine;
    var cell = this.currentCell();
    var frames = cell.frames;
    var cellLength = frames.length;
    line.setSelectionElem(1,cellLength);
};


/* delete window
===================================*/
Editor.prototype.showDeleteWindow = function(){
    SoundManager.playOk();

    var window;
    if(this._timingMode){
        window = new Window_DeleteTiming();
        window.setHandler('timing', this.didSelectDeleteTiming.bind(this));
    }else{
        window = new Window_Delete();
        window.setHandler('clear', this.didSelectClearFrame.bind(this));
        window.setHandler('frame', this.didSelectDeleteFrame.bind(this));
        window.setHandler('clearCell', this.didSelectClearCell.bind(this));
        window.setHandler('cell', this.didSelectDeleteCell.bind(this));
    }
    window.setHandler('cancel', this.didCancelDelete.bind(this));

    this._activeWindow = window;
    this.addChild(window);
    window.setup();
};
Editor.prototype.didSelectClearFrame = function(){
    SoundManager.playOk();
    this.clearFrame();
    this._activeWindow.close();
};
Editor.prototype.didSelectDeleteFrame = function(){
    SoundManager.playOk();
    this.deleteCurrentFrame();
    this._activeWindow.close();
};
Editor.prototype.didSelectClearCell = function(){
    SoundManager.playOk();
    this.clearCell();
    this._activeWindow.close();
};
Editor.prototype.didSelectDeleteCell = function(){
    SoundManager.playOk();
    this.deleteCurrentCell();
    this._activeWindow.close();
};
Editor.prototype.didCancelDelete = function(){
    SoundManager.playCancel();
    this._activeWindow.close();
};
Editor.prototype.didSelectDeleteTiming = function(){
    SoundManager.playOk();
    this.deleteCurrentTiming();
    this._activeWindow.close();
}



//=============================================================================
// Window_EditorBase
//=============================================================================
var Window_EditorBase;
if(isMZ)(()=>{
    Window_EditorBase = Window_Command;
})();
if(!isMZ)(()=>{
    Window_EditorBase = function Window_EditorBase(){
        this.initialize.apply(this, arguments);
    };
    Window_EditorBase.prototype = Object.create(Window_Command.prototype);
    Window_EditorBase.prototype.constructor = Window_EditorBase;
    Window_EditorBase.prototype.initialize = function(x,y,width){
        this._width = width;
        Window_Command.prototype.initialize.call(this,...arguments);
    };
    Window_EditorBase.prototype.windowWidth = function(){
        return this._width;
    }
})();


//=============================================================================
// Window_Delete
//=============================================================================
function Window_Delete() {
    this.initialize(...arguments);
};
Window_Delete.prototype = Object.create(Window_EditorBase.prototype);
Window_Delete.prototype.constructor = Window_Delete;
Window_Delete.prototype.initialize = function(){
    var lines = Window_Delete.COMMANDS.length;
    var width = 462;
    var height = this.fittingHeight(lines);

    var x = (Graphics.width-width)/2;
    var y = (Graphics.height-height)/2;
    if(isMZ){
        var rect = new Rectangle(x,y,width,height)
        Window_EditorBase.prototype.initialize.call(this, rect);
    }else{
        Window_EditorBase.prototype.initialize.call(this,x,y,width,height);
    }

    this.openness = 0;
    this.deactivate();
};
Window_Delete.COMMANDS = [
    ['現在のモーションをクリア','clear'],
    ['現在のモーションを削除','frame'],
    ['現在のセルをクリア','clearCell'],
    ['現在のセルを削除', 'cell'],
    ['キャンセル','cancel'],
];
Window_Delete.prototype.makeCommandList = function() {
    for(const command of Window_Delete.COMMANDS){
        this.addCommand(command[0],command[1]);      
    }
};
Window_Delete.prototype.setup = function() {
    this.refresh();
    if(isMZ){
        this.forceSelect(0);
    }
    this.activate();
    this.open();
};

//=============================================================================
// Window_DeleteTiming
//=============================================================================
function Window_DeleteTiming() {
    this.initialize(...arguments);
};
Window_DeleteTiming.prototype = Object.create(Window_EditorBase.prototype);
Window_DeleteTiming.prototype.constructor = Window_DeleteTiming;
Window_DeleteTiming.prototype.initialize = function(){
    var lines = Window_DeleteTiming.COMMANDS.length;
    var width = 462;
    var height = this.fittingHeight(lines);
    var x = (Graphics.width-width)/2;
    var y = (Graphics.height-height)/2;
    if(isMZ){
        var rect = new Rectangle(x,y,width,height)
        Window_EditorBase.prototype.initialize.call(this, rect);
    }else{
        Window_EditorBase.prototype.initialize.call(this,x,y,width,height);
    }
    
    this.openness = 0;
    this.deactivate();
};
Window_DeleteTiming.COMMANDS = [
    ['編集中のタイミングデータを削除','timing'],
    ['キャンセル','cancel'],
];
Window_DeleteTiming.prototype.makeCommandList = function() {
    for(const command of Window_DeleteTiming.COMMANDS){
        this.addCommand(command[0],command[1]);      
    }
};
Window_DeleteTiming.prototype.setup = function() {
    this.refresh();
    this.forceSelect(0);
    this.activate();
    this.open();
};






/* edit mode
===================================*/
Editor.prototype.updateInputEditMode = function(){
    var shift = Input.isPressed('shift');
    if(parameters.reverseShift){
        shift = !shift;
    }

    if(Input.isTriggered(backspaceKey)){
        this.showDeleteWindow();
    }else if(Input.isRepeated('up')){
        if(Input.isPressed('control')){
            this.plusScaleY(shift);
        }else{
            this.minusPosY(shift);
        }
    }else if(Input.isRepeated('down')){
        if(Input.isPressed('control')){
            this.minusScaleY(shift);
        }else{
            this.plusPosY(shift);
        }
    }else if(Input.isRepeated('left')){
        if(Input.isPressed('control')){
            this.minusScaleX(shift);
        }else{
            this.minusPosX(shift);
        }
    }else if(Input.isRepeated('right')){
        if(Input.isPressed('control')){
            this.plusScaleX(shift);
        }else{
            this.plusPosX(shift);
        }

    }else if(Input.isRepeated('pagedown')){
        if(Input.isPressed('control')){
            this.plusOpacity(shift)
        }else{
            this.plusAngle(shift);
        }
    }else if(Input.isRepeated('pageup')){
        if(Input.isPressed('control')){
            this.minusOpacity(shift)
        }else{
            this.minusAngle(shift);
        }
    }
};

Editor.prototype.currentAnimationCell = function(){
    var sprite = this._animationSprite;
    return sprite._cellSprites[this._cellIndex];
};
Editor.prototype.clearFrame = function(){
    SoundManager.playCancel();

    var frame = null;
    if(this._frameIndex===0){
        frame = this.createFrame(0);
    }else{
        frame = JsonEx.makeDeepCopy(this.currentCell().frames[this._frameIndex-1]);
    }

    this.currentCell().frames[this._frameIndex] = frame;
    this.refreshFrameLine();
    this.selectFrame(this._frameIndex,true);
};
Editor.prototype.clearCell = function(){
    SoundManager.playCancel();

    var cell = this.createCell(0);

    this._animation.cells[this._cellIndex] = cell;

    this.refreshCellLine();
    this.selectCell(this._cellIndex,true);
};
Editor.prototype.plusScaleX = function(shift=false){
    var delta = (shift ? 0.1 : 0.01);
    this._plusScaleX(delta);
};
Editor.prototype.minusScaleX = function(shift=false){
    var delta = -(shift ? 0.1 : 0.01);
    this._plusScaleX(delta);
};


Editor.prototype._plusScaleX = function(delta){
    var data = this.currentFrame().data;
    var rate = 0;
    var scaleX = data[IDXES.scaleX];
    var scaleY = data[IDXES.scaleY];
    if(scaleX===0){
        if(scaleY===0){
            rate = 1;
        }
    }else{
        rate = scaleY/scaleX;
    }
    data[IDXES.scaleX] += delta;

    var cell = this.currentAnimationCell();
    cell.scale.x = data[IDXES.scaleX];

    if(this._scaleRateFix&&rate){
        cell.scale.y = data[IDXES.scaleY] = rate*data[IDXES.scaleX];
    }
    this.refreshScaleLineValues();
};
Editor.prototype.plusScaleY = function(shift=false){
    var delta = (shift ? 0.1 : 0.01);
    this._plusScaleY(delta);
};
Editor.prototype.minusScaleY = function(shift=false){
    var delta = -(shift ? 0.1 : 0.01);
    this._plusScaleY(delta);
};
Editor.prototype._plusScaleY = function(delta){
    var data = this.currentFrame().data;
    var rate = 0;
    var scaleX = data[IDXES.scaleX];
    var scaleY = data[IDXES.scaleY];
    if(scaleY===0){
        if(scaleX===0){
            rate = 1;
        }
    }else{
        rate = scaleX/scaleY;
    }
    data[IDXES.scaleY] += delta;

    var cell = this.currentAnimationCell();
    cell.scale.y = data[IDXES.scaleY];

    if(this._scaleRateFix&&rate){
        cell.scale.x = data[IDXES.scaleX] = rate*data[IDXES.scaleY];
    }
    this.refreshScaleLineValues();
};
Editor.prototype.plusPosX = function(shift=false){
    var delta = (shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.x] += delta;
    this.currentAnimationCell().x = data[IDXES.x];
    this.refreshPosLineValues();
};
Editor.prototype.minusPosX = function(shift=false){
    var delta = -(shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.x] += delta;
    this.currentAnimationCell().x = data[IDXES.x];
    this.refreshPosLineValues();
};
Editor.prototype.plusPosY = function(shift=false){
    var delta = (shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.y] += delta;
    this.currentAnimationCell().y = data[IDXES.y];
    this.refreshPosLineValues();
};
Editor.prototype.minusPosY = function(shift=false){
    var delta = -(shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.y] += delta;
    this.currentAnimationCell().y = data[IDXES.y];
    this.refreshPosLineValues();
};
Editor.prototype.plusOpacity = function(shift=false){
    var delta = (shift ? 25.5 : 2.55);
    var data = this.currentFrame().data;
    data[IDXES.opacity] = (data[IDXES.opacity]+delta).clamp(0,255);
    this.currentAnimationCell().opacity = data[IDXES.opacity];
    this.refreshOpacityLineValues();
};
Editor.prototype.minusOpacity = function(shift=false){
    var delta = -(shift ? 25.5 : 2.55);
    var data = this.currentFrame().data;
    data[IDXES.opacity] = (data[IDXES.opacity]+delta).clamp(0,255);
    this.currentAnimationCell().opacity = data[IDXES.opacity];

    this.refreshOpacityLineValues();
};
Editor.prototype.plusAngle = function(shift=false){
    var delta = (shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.angle] += delta;
    this.currentAnimationCell().rotation = data[IDXES.angle]*Math.PI/180;
    this.refreshAngleLineValues();
};
Editor.prototype.minusAngle = function(shift=false){
    var delta = -(shift ? 10 : 1);
    var data = this.currentFrame().data;
    data[IDXES.angle] += delta;
    this.currentAnimationCell().rotation = data[IDXES.angle]*Math.PI/180;
    this.refreshAngleLineValues();
};



/* end
===================================*/
Editor.prototype.showEndWindow = function(){
    SoundManager.playOk();
    var window = new Window_End();
    this._activeWindow = window;
    window.setHandler('save', this.didSelectSaveAndEnd.bind(this));
    window.setHandler('end', this.didSelectEnd.bind(this));
    window.setHandler('cancel', this.didCancelEnd.bind(this));
    this.addChild(window);
    window.setup();
};
Editor.prototype.didSelectSaveAndEnd = function(){
    SoundManager.playSave();
    this.didClickSaveButton()
    this._activeWindow.close();
    this.processEnd();
};
Editor.prototype.didSelectEnd = function(){
    SoundManager.playOk();
    this.processEnd();
    this._activeWindow.close();
};
Editor.prototype.didCancelEnd = function(){
    SoundManager.playCancel();
    this._activeWindow.close();
};
Editor.prototype.processEnd = function(){
    var data = this._animation;
    this._end = true;

    this.resignKeyListeners();
};






//=============================================================================
// Window_End
//=============================================================================
function Window_End() {
    this.initialize(...arguments);
};
Window_End.COMMANDS = [
    ['保存してから終了','save'],
    ['そのまま終了', 'end'],
    ['キャンセル','cancel'],
];
Window_End.prototype = Object.create(Window_EditorBase.prototype);
Window_End.prototype.constructor = Window_End;
Window_End.prototype.initialize = function(){
    var lines = Window_End.COMMANDS.length;
    var width = 462;
    var height = this.fittingHeight(lines);
    var x = (Graphics.width-width)/2;
    var y = (Graphics.height-height)/2;

    if(isMZ){
        var rect = new Rectangle(x,y,width,height)
        Window_EditorBase.prototype.initialize.call(this, rect);
    }else{
        Window_EditorBase.prototype.initialize.call(this,x,y,width,height);
    }
    this.openness = 0;
    this.deactivate();
};
Window_End.prototype.makeCommandList = function() {
    for(const command of Window_End.COMMANDS){
        this.addCommand(command[0],command[1]);      
    }
};
Window_End.prototype.setup = function() {
    this.refresh();
    if(isMZ){
        this.forceSelect(0);
    }
    this.activate();
    this.open();
};





/* center point
===================================*/
Editor.prototype.createCenterSprite = function(){
    var bitmap = new Bitmap(16,16);
    var color = 'rgba(255,0,0,0.6)';
    bitmap.fillRect(7,0,2,16,color);
    bitmap.fillRect(0,7,16,2,color);

    var sprite = new Sprite(bitmap);
    sprite.anchor.set(0.5,0.5);
    sprite.visible = false;
    this._centerSprite = sprite;
    this._animationSprite.addChild(sprite);
};
Editor.prototype.updateCenterSprite = function(){
    var cellSprite = this.currentAnimationCell();
    var sprite = this._centerSprite;
    sprite.x = cellSprite.x;
    sprite.y = cellSprite.y;
};




/* editor line
===================================*/
Editor.SELECTION_WIDHT = 24;
Editor.prototype.createLines = function(){
    var line;
    var idx = 0;
    var buttonW = 120;

    line = new EditorLine(idx++);
    this._headerLine = line;
    this.addChild(line);
    line.addElem('id','ID');
    line.addElem('name','名前[N]',buttonW+20,false,this.didClickNameButton.bind(this));
    line.addElem('pos','位置[P]',buttonW-20,false,this.didClickPosition.bind(this));
    line.setValue('id',this._animation.id+'<'+(Number(this._animation.id)+Number(parameters.animationId)-1)+'>');

    idx += 1;

    var w = Editor.SELECTION_WIDHT;


    line = new EditorLine(idx++,this.didSelectCellLine.bind(this));
    this._cellLine = line;
    this.addChild(line);
    line.addElem('header','セル[C]：');
    line.addElem('viewMode','表示[V]',buttonW,false,this.didClickViewMode.bind(this));

    line = new EditorLine(idx++);
    this._imageLine = line;
    this.addChild(line);
    line.addElem('header','セル設定：');
    line.addElem('image','画像[I]',buttonW,false,this.didClickImageButton.bind(this));
    line.addElem('blend','ブレンド[B]:',buttonW,false,this.didClickBlendButton.bind(this));

    line = new EditorLine(idx++,this.didSelectFrameLine.bind(this));
    this._frameLine = line;
    this.addChild(line);
    line.addElem('header','モーション[M]：');

    idx += 1;

    line = new EditorLine(idx++);
    this._editHeaderLine = line;
    this.addChild(line);
    line.addElem('header','【モーション設定】');
    line.refresh();

    line = new EditorLine(idx++);
    this._motionLine = line;
    this.addChild(line);
    line.addElem('duration','デュレーション[D]',buttonW,false,this.didClickFrameButton.bind(this));
    line.addElem('easing','イージング[E]',buttonW,false,this.didClickEasingButton.bind(this));

    line = new EditorLine(idx++);
    this._posLine = line;
    this.addChild(line);
    line.addElem('pos','位置[矢印]');
    line.addElem('x','x');
    line.addElem('y','y');

    line = new EditorLine(idx++);
    this._scaleLine = line;
    this.addChild(line);
    line.addElem('scale','拡大率[ctrl+矢印]');
    line.addElem('scaleX','横');
    line.addElem('scaleY','縦');
    line.addElem('fixScaleRate','比率[L]',buttonW-30,false,this.didClickFixScaleRateButton.bind(this));

    line = new EditorLine(idx++);
    this._angleLine = line;
    this.addChild(line);
    line.addElem('angle','角度[Q/W]');

    line = new EditorLine(idx++);
    this._opacityLine = line;
    this.addChild(line);
    line.addElem('opacity','不透明度[ctrl+Q/W]');

    line = new EditorLine(idx++);
    this._animationLine = line;
    this.addChild(line);
    line.addElem('animation','アニメ[A]',buttonW,false,this.didClickAnimationButton.bind(this));
    line.addElem('speed','スピード');
    line.addElem('loop','ループ');
};
Editor.prototype.refreshAllLines = function(){
    var line = this._headerLine;
    this.refreshHeaderLineValues(true);
    line.refresh();

    line = this._cellLine;
    this.refreshCellViewMode(true);
    this.refreshCellLine();

    line = this._frameLine;
    line.refresh();
};

Editor.prototype.refreshHeaderLineValues = function(noRefresh=false){
    var line = this._headerLine;
    var value='---';
    switch(this._animation.position){
    case 0:
        value = '頭上';
        break;
    case 1:
        value = '中心';
        break;
    case 2:
        value = '足元';
        break;
    case 3:
        value = '全体';
        break;
    }
    line.setValue('pos',value,noRefresh);


    var name = this.animationDBData().name;
    if(!name){
        name = '-未設定-';
    }
    line.setValue('name',name);
};

Editor.prototype.didClickNameButton = function(){
    if(!parameters.enableWriteAnimation){
        SoundManager.playBuzzer();
        this.showTempGuide(['名前の変更を行うにはプラグイン設定より','「Animation.json上書き有効化」をONにする必要があります。']);
        return;
    }

    SoundManager.playCursor();
    this.setMode('editName');

    this.animationDBData().name = ' ';
    this.refreshHeaderLineValues();
};
Editor.prototype.startEditName = function(){
    var name = this.animationDBData().name;
    if(!name){
        name = '-未設定-';
    }
    this._nameInput = new EditorNameInput(name,this.didChangeNameInput.bind(this));

    var rect = new Rectangle(10,100,500,420);
    this._inputWindow = new Window_NameInput(rect);
    this._inputWindow.setEditWindow(this._nameInput);
    this._inputWindow.setHandler('ok',this.didEditNameInput.bind(this));
    this.addChild(this._inputWindow);
};
Editor.prototype.didChangeNameInput = function(name){
    this.animationDBData().name = name;
    this.refreshHeaderLineValues();
};
Editor.prototype.didEditNameInput = function(){
    this._inputWindow.close();
};
Editor.prototype.endNameInput = function(){
    this.removeChild(this._inputWindow);
    this._inputWindow.destroy();
    this._inputWindow = null;

    this._nameInput.releaseHandler();
    this._nameInput = null;
};
Editor.prototype.didClickPosition = function(){
    SoundManager.playCursor();
    this._animation.position += 1;
    if(this._animation.position>3){
        this._animation.position = 0;
    }
    this.refreshHeaderLineValues();
    this._headerLine.refresh();

    this._animationSprite.updatePosition();
};



/* cell
===================================*/
Editor.prototype.didSelectCellLine = function(index){
    SoundManager.playCursor();
    this.selectCell(index);
};
Editor.prototype.selectCell = function(index,force=false){
    var cells = this._animation.cells;
    if(index<0){
        index = cells.length-1;
    }else if(index>cells.length-1){
        index = 0;
    }
    if(!force && this._cellIndex===index)return;
    this._cellIndex = index;

    var line = this._cellLine;
    line.select(index,force);

    this.refreshViewModeOpacityRate();

    this.refreshImageLine();

    this.refreshFrameLine();
    var frames = this.currentCell().frames;
    var index = this._frameIndex.clamp(0,frames.length-1);
    this.selectFrame(0,true);
};
Editor.prototype.refreshCellViewMode = function(noRefresh=false){
    var line = this._cellLine;
    var value='---';
    switch(this._viewMode){
    case 0:
        value = '全セル';
        break;
    case 1:
        value = '半透明';
        break;
    case 2:
        value = 'ソロ';
        break;
    }
    line.setValue('viewMode',value,noRefresh);
};
Editor.prototype.didClickViewMode = function(){
    SoundManager.playCursor();

    this._viewMode += 1;
    if(this._viewMode >= 3){
        this._viewMode = 0;
    }

    this.refreshCellViewMode();
    this._cellLine.refresh();

    this.refreshViewModeOpacityRate();
};


/* frame
===================================*/
Editor.prototype.didSelectFrameLine = function(index){
    SoundManager.playCursor();
    this.selectFrame(index);
};
Editor.prototype.selectFrame = function(index,force=false){
    var frames = this.currentCell().frames;
    if(index<0){
        index = frames.length-1;
    }else if(index>frames.length-1){
        index = 0;
    }

    if(!force&&this._frameIndex===index)return;
    this._frameIndex = index;

    var line = this._frameLine;
    line.select(index,force);

    this.refreshAnimationSpriteFrame();


    this.refreshMotionLine(true);
    this._motionLine.refresh();

    this.refreshPosLineValues();
    this.refreshScaleLineValues();
    this.refreshAngleLineValues();
    this.refreshOpacityLineValues();
    this.refreshAnimationLine();
};

/* edit motion
===================================*/
Editor.prototype.didClickFrameButton = function(){
    SoundManager.playCursor();

    var line = this._motionLine;
    var symbol = 'duration';
    var handler = this.setFrame.bind(this);
    var def = 10;
    var min = 0;
    var max = 1000000;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setFrame = function(value){
    var frame = this.currentFrame();
    frame.duration = value;
};


/* imageLine
===================================*/
Editor.prototype.refreshImageLine = function(noRefresh=false){
    var line = this._imageLine;
    var cell = this.currentCell();
    
    var value = cell.image.replace('ANIM:','');
    line.setValue('image',value,true);

    switch(cell.blend){
    case 1: 
        value = '加算';
        break;
    case 2: 
        value = '乗算';
        break;
    case 3: 
        value = 'スクリーン';
        break;
    default:
        value = '通常';
        break;
    }
    line.setValue('blend',value,true);

    if(!noRefresh){
        line.refresh();
    }
};

Editor.prototype.didClickImageButton = function(){
    this.setMode('editImage');
};
Editor.prototype.startPickingImage = function(){
    if(!this._imagePicker){
        this.createImagePicker();   
        if(!this._imagePicker)return;
    }

    var image = this.currentCell().image;
    this._imagePicker.startPicking(this,image);
    this.startPicking(this._imagePicker);
};
Editor.prototype.createImagePicker = function(){
    var picker = new ImagePicker();
    this._imagePicker = picker;
    this.addChild(picker);
    picker.x = 10;
    picker.y = 0;
};
Editor.prototype.setImage = function(image){
    var cell = this.currentCell();
    cell.image = image;
    this._imageLine.setValue('image',image);
};
Editor.prototype.setImageMirror = function(){
    var cell = this.currentCell();
    cell.imageMirror = !cell.imageMirror;
};

Editor.prototype.didClickBlendButton = function(){
    SoundManager.playCursor();

    var cell = this.currentCell();
    if(cell.blend===undefined){
        cell.blend = 0;
    }else{
        cell.blend += 1;
        if(cell.blend>=4){
            cell.blend = 0;
        }
    }
    this.refreshImageLine();
};



/* animationLine
===================================*/
Editor.prototype.refreshAnimationLine = function(noRefresh=false){
    var line = this._animationLine;
    var frame = this.currentFrame();
    var imageData = frame.data[IDXES.animation];

    var value ;
    if(!imageData){
        value = '---'
    }else{
        value = imageData.images.length+'コマ';
    }
    line.setValue('animation',value,noRefresh);

    if(imageData){
        value = imageData.speed;
        line.setEnabled('speed');
        line.setValue('speed',value,noRefresh);
    }else{
        line.setDisabled('speed');
    }

    if(imageData){
        value = imageData.loop ? 'ON' : 'OFF';
        line.setEnabled('loop');
        line.setValue('loop',value,noRefresh);
    }else{
        line.setDisabled('loop');
    }
    line.refresh();
};
Editor.prototype.didClickAnimationButton = function(){
    this.setMode('editAnimation');
};
Editor.prototype.didClickSpeedButton = function(){
    this.setMode('editAnimationSpeed');
};
Editor.prototype.didClickLoopButton = function(){
    var imageData = this.currentFrame().data[IDXES.animation];
    if(!imageData){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playCursor();
    imageData.loop = !imageData.loop;
    this.refreshAnimationLine();
};

Editor.prototype.startPickingAnimation = function(){
    if(!this._animationPicker){
        this.createAnimationPicker();   
        if(!this._animationPicker)return;
    }

    this.refreshAnimationSpriteFrame();

    var imageData = this.currentFrame().data[IDXES.animation];
    this._animationPicker.startPicking(this,imageData);
    this.startPicking(this._animationPicker);
};
Editor.prototype.createAnimationPicker = function(){
    var picker = new AnimationPicker();
    this._animationPicker = picker;
    this.addChild(picker);
    picker.x = 10;
    picker.y = 0;
};
Editor.prototype.setAnimation = function(images,speed=4,loop=false){
    var frame = this.currentFrame();
    if(!images || images.length === 0){
        frame.data[IDXES.animation] = null;
    }else{
        frame.data[IDXES.animation] = {
            images:images,
            speed:speed,
            loop:loop,
            mirror:false
        }
    }
    this.refreshAnimationLine();
};
Editor.prototype.setAnimationMirror = function(){
    var frame = this.currentFrame();
    var animation = frame.data[IDXES.animation];
    if(animation){
        animation.mirror = !animation.mirror;
    }

    this.currentAnimationCell().scale.x*=-1;
};
Editor.prototype.currentFrameImage = function(){
    var cell = this.currentCell();
    var frames = cell.frames;
    var length = frames.length;
    var image = cell.image;

    for(var i = 0; i<this._frameIndex; i=(i+1)|0){
        var frame = frames[i];
        var imageData = frame.data[IDXES.animation];
        if(!imageData)continue;
        if(imageData.loop){
            image = imageData.images[0];
        }else{
            image = imageData.images[imageData.images.length-1];
        }
    }
    return image;
};


/* motionLine
===================================*/
Editor.prototype.refreshMotionLine = function(noRefresh=false){
    var line = this._motionLine;
    var frame = this.currentFrame();
    
    var value = frame.duration;
    line.setValue('duration',value,noRefresh);

    var value = frame.easing;
    line.setValue('easing',value,noRefresh);
};

Editor.prototype.didClickEasingButton = function(){
    this.setMode('editEasing');
};
Editor.prototype.startPickingEasing = function(){
    if(!this._easingPicker){
        this.createEasingPicker();   
        if(!this._easingPicker)return;
    }

    var easing = this.currentFrame().easing;
    this._easingPicker.startPicking(this,easing);
    this.startPicking(this._easingPicker);
};
Editor.prototype.createEasingPicker = function(){
    var picker = new EasingPicker();
    this._easingPicker = picker;
    this.addChild(picker);
    picker.x = 10;
    picker.y = 0;
};
Editor.prototype.setEasing = function(easing){
    var frame = this.currentFrame();
    frame.easing = easing;
    this._motionLine.setValue('easing',easing);
};


/* infoLines
===================================*/
Editor.prototype.refreshPosLineValues = function(){
    var line = this._posLine;
    var data = this.currentFrame().data;
    line.setValue('x',Math.round(data[IDXES.x]));
    line.setValue('y',Math.round(data[IDXES.y]));
    line.refresh();
};
Editor.prototype.refreshScaleLineValues = function(){
    var line = this._scaleLine;
    var data = this.currentFrame().data;
    line.setValue('scaleX',Math.round(data[IDXES.scaleX]*100)+'%');
    line.setValue('scaleY',Math.round(data[IDXES.scaleY]*100)+'%');
    line.setValue('fixScaleRate',this._scaleRateFix?'固定':'可変');
    line.refresh();
};
Editor.prototype.didClickFixScaleRateButton = function(){
    this._scaleRateFix = !this._scaleRateFix;
    this.refreshScaleLineValues();
    SoundManager.playCursor();
};
Editor.prototype.refreshAngleLineValues = function(){
    var line = this._angleLine;
    var value = this.currentFrame().data[IDXES.angle];
    line.setValue('angle',Math.round(value)+'度');
    line.refresh();
};
Editor.prototype.refreshOpacityLineValues = function(){
    var line = this._opacityLine;
    var value = this.currentFrame().data[IDXES.opacity];
    line.setValue('opacity',Math.round(value));
    line.refresh();
};



/* timing mode
===================================*/
Editor.prototype.startTimingMode = function(){
    if(this._timingMode)return;

    SoundManager.playOk();
    if(!this._timingHeaderLine){
        this.createTimingLines();
    }

    this._switchTimingMode();

    this.selectTiming(0);
    this.setMode('timing');
};
Editor.prototype.endTimingMode = function(){
    if(!this._timingMode)return;

    this._switchTimingMode();
    this.setMode('edit');
};
Editor.prototype._switchTimingMode = function(){
    var lines = this.allLines();
    for(const line of lines){
        this.removeChild(line);   
    }

    this._timingMode = !this._timingMode;

    this.refreshViewModeOpacityRate();

    lines = this.allLines();
    for(const line of lines){
        this.addChild(line);  
    }
};
Editor.prototype.createTimingLines = function(){
    var line;
    var idx = 0;
    var buttonW = 120;

    line = new EditorLine(idx++);
    this._timingHeaderLine = line;
    this.addChild(line);
    line.addElem('header','【タイミング設定】');
    line.refresh();

    var w = Editor.SELECTION_WIDHT;

    line = new EditorLine(idx++,this.didSelectTimingLine.bind(this));
    this._timingLine = line;
    this.addChild(line);
    line.addElem('header','タイミングデータ[T]：');

    idx += 0.5;

    line = new EditorLine(idx++);
    this._timingFrameLine = line;
    this.addChild(line);
    line.addElem('timing','発生フレーム[F]',buttonW+50,false,this.didClickTimingFrameButton.bind(this));

    idx += 0.5;

    line = new EditorLine(idx++);
    this._seHeaderLine = line;
    this.addChild(line);
    line.addElem('header','【SE設定】');
    line.refresh();

    line = new EditorLine(idx++);
    this._seLine = line;
    this.addChild(line);
    line.addElem('se','SE名[N]',buttonW+50,false,this.didClickSeButton.bind(this));
    line.addElem('volume','ボリューム[V]',buttonW,false,this.didClickSeVolumeButton.bind(this));

    line = new EditorLine(idx++);
    this._seSettingLine = line;
    this.addChild(line);
    line.addElem('pitch','ピッチ[P]',buttonW,false,this.didClickSePitchButton.bind(this));
    line.addElem('pan','パン[A]',buttonW,false,this.didClickSePanButton.bind(this));

    idx += 0.5;

    line = new EditorLine(idx++);
    this._flashHeaderLine = line;
    this.addChild(line);
    line.addElem('header','【フラッシュ設定】');
    line.refresh();

    line = new EditorLine(idx++);
    this._flashLine = line;
    this.addChild(line);
    line.addElem('flash','フラッシュ色[C]',buttonW+100,false,this.didClickFlashButton.bind(this));
    line.addElem('flashOpacity','不透明度[O]',buttonW,false,this.didClickFlashOpacityButton.bind(this));

    line = new EditorLine(idx++);
    this._flashSettingLine = line;
    this.addChild(line);
    line.addElem('flashDuration','デュレーション[D]',buttonW,false,this.didClickFlashDurationButton.bind(this));
    line.addElem('flashScope','対象範囲[S]',buttonW,false,this.didClickFlashScopeButton.bind(this));
};


Editor.prototype.didSelectTimingLine = function(index){
    SoundManager.playCursor();
    this.selectTiming(index);
};
Editor.prototype.selectTiming = function(index,force=false){
    var timings = this.timingData();
    if(index<0){
        index = timings.length-1;
    }else if(index>timings.length-1){
        index = 0;
    }

    if(!force&&this._timingIndex===index)return;
    this._timingIndex = index;

    var line = this._timingLine;
    line.select(index,force);

    this.refreshTimingLine(true);

    this.refreshTimingFrameLine();
    this.refreshSeLineValues();
    this.refreshFlashLineValues();
};
Editor.prototype.refreshTimingLine = function(){
    var line = this._timingLine;
    var timings = this.timingData();
    var length = timings.length;
    line.setSelectionElem(1,length);
};
Editor.prototype.addTiming = function(index,timings=this.timingData()){
    if(index<0 || index>timings.length){
        SoundManager.playBuzzer();
        return;
    }

    SoundManager.playOk();

    var timing = this.createTiming();
    timings.splice(index,0,timing);

    this.refreshTimingLine();
    this.selectTiming(index,true);
};
// ,"timings":[{"flashColor":[255,255,255,255],"flashDuration":5,"flashScope":0,"frame":3,"se":{"name":"Wind2","pan":0,"pitch":130,"volume":40}}]



Editor.prototype.createTiming = function(idx=0){
    return {
        flashColor:[255,255,255,255],
        flashDuration:30,
        flashScope:2,
        frame:0,
        se:{
            name:'Blow1',
            pan:0,
            pitch:100,
            volume:90
        }
    }
};
Editor.prototype.deleteCurrentTiming = function(){
    var timings = this.timingData();
    if(timings.length===1){
        this.clearTiming();
        return;
    }

    SoundManager.playCancel();
    timings.splice(this._timingIndex,1);

    this.refreshTimingLine();

    var index = this._timingIndex.clamp(0,timings.length-1);
    this.selectTiming(index,true);
};
Editor.prototype.clearTiming = function(){
    SoundManager.playCancel();

    var timings = this.timingData();
    timings[this._timingIndex] = this.createTiming();

    this.refreshTimingLine();
    this.selectTiming(this._timingIndex,true);
};

Editor.prototype.didClickTimingFrameButton = function(){
    SoundManager.playCursor();

    var line = this._timingFrameLine;
    var symbol = 'timing';
    var handler = this.setTimingFrame.bind(this);
    var def = 0;
    var min = 0;
    var max = 1000000;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setTimingFrame = function(value){
    var frame = this.currentTiming();
    frame.frame = value;
};
Editor.prototype.refreshTimingFrameLine = function(){
    var value = this.currentTiming().frame;

    var line = this._timingFrameLine;
    line.setValue('timing',value);
    line.refresh();    
};


Editor.prototype.refreshSeLineValues = function(){
    var value;
    var line = this._seLine;

    var se = this.currentTiming().se;


    //seName
    value = se.name;
    line.setValue('se',value);

    //volume
    value = se.volume;
    line.setValue('volume',Math.round(value));
    line.refresh();


    line = this._seSettingLine;

    //pitch
    value = se.pitch;
    line.setValue('pitch',Math.round(value));

    //pan
    value = se.pan;
    line.setValue('pan',Math.round(value));

    line.refresh();
};
Editor.prototype.didClickSeButton = function(){
    this.setMode('editSe');
};
Editor.prototype.startPickingSe = function(){
    if(!this._sePicker){
        this.createSePicker();   
        if(!this._sePicker)return;
    }

    var se = this.currentFrame().se;
    var name = se ? se.name : '';
    this._sePicker.startPicking(this,name);
    this.startPicking(this._sePicker);
};
Editor.prototype.createSePicker = function(){
    var picker = new SePicker();
    this._sePicker = picker;
    this.addChild(picker);
    picker.x = 10;
    picker.y = 0;
};
Editor.prototype.setSeName = function(name){
    var se = this.currentTiming().se;
    se.name = name;
    this.refreshSeLineValues();
};
Editor.prototype.didClickSeVolumeButton = function(){
    SoundManager.playCursor();

    var line = this._seLine;
    var symbol = 'volume';
    var handler = this.setSeVolume.bind(this);
    var def = 90;
    var min = 0;
    var max = 100;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setSeVolume = function(value){
    var se = this.currentTiming().se;
    se.volume = value;
};
Editor.prototype.didClickSePitchButton = function(){
    SoundManager.playCursor();

    var line = this._seSettingLine;
    var symbol = 'pitch';
    var handler = this.setSePitch.bind(this);
    var def = 100;
    var min = 10;
    var max = 1000;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setSePitch = function(value){
    var se = this.currentTiming().se;
    se.pitch = value;
};
Editor.prototype.didClickSePanButton = function(){
    SoundManager.playCursor();

    var line = this._seSettingLine;
    var symbol = 'pan';
    var handler = this.setSePan.bind(this);
    var def = 0;
    var min = -100;
    var max = 100;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setSePan = function(value){
    var se = this.currentTiming().se;
    se.pan = value;
};


/* flash
===================================*/
Editor.prototype.didClickFlashButton = function(){
    this.setMode('editFlash');
    SoundManager.playCursor();
};
Editor.prototype.startPickingFlash = function(){
    if(!this._colorPicker){
        this.createColorPicker();   
        if(!this._colorPicker)return;
    }

    var color = this.currentTiming().flashColor;
    this._colorPicker.startPicking(this,color)
    this._colorPicker.visible = true;
    this.startPicking(this._colorPicker);
};
Editor.prototype.createColorPicker = function(){
    var size = 144;
    var picker = new ColorPicker(size);
    this._colorPicker = picker;
    this.addChild(picker);
    picker.x = 10;
    picker.y = 30;
    picker.visible = false;
};
Editor.prototype.setColor = function(r,g,b){
    var color = this.currentTiming().flashColor;
    color[0] = r;
    color[1] = g;
    color[2] = b;

    this.refreshFlashLineValues();
};

Editor.prototype.didClickFlashOpacityButton = function(){
    SoundManager.playCursor();

    var line = this._flashLine;
    var symbol = 'flashOpacity';
    var handler = this.setFlashOpacity.bind(this);
    var def = 255;
    var min = 0;
    var max = 255;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setFlashOpacity = function(value){
    var timing = this.currentTiming();
    timing.flashColor[3] = value;
};

Editor.prototype.didClickFlashDurationButton = function(){
    SoundManager.playCursor();

    var line = this._flashSettingLine;
    var symbol = 'flashDuration';
    var handler = this.setFlashDuration.bind(this);
    var def = 30;
    var min = 0;
    var max = 999999;
    this.startInputtingMode(line,symbol,handler,def,min,max);
};
Editor.prototype.setFlashDuration = function(value){
    var timing = this.currentTiming();
    timing.flashDuration = value;
};
Editor.prototype.didClickFlashScopeButton = function(){
    SoundManager.playCursor();

    var timing = this.currentTiming();
    var scope;
    if(timing.flashScope!==undefined){
        scope = timing.flashScope+1;
        if(scope>=Editor.FLASH_SCOPE_NAMES.length){
            scope = 0;
        }
    }else{
        scope = 0;
    }
    timing.flashScope = scope;
    this.refreshFlashLineValues();
};
Editor.FLASH_SCOPE_NAMES = [
    'なし','対象','画面','対象消去'
];

Editor.prototype.refreshFlashLineValues = function(){
    var timing = this.currentTiming();
    var line = this._flashLine;
    var value;

    //flash color
    var color = timing.flashColor.concat()
    color.pop();
    value = String(color)
    line.setValue('flash',value);

    value = String(timing.flashColor[3]);
    line.setValue('flashOpacity',value);


    var line = this._flashSettingLine;

    // flash duration
    value = timing.flashDuration;
    line.setValue('flashDuration',value);

    //scope
    value = Editor.FLASH_SCOPE_NAMES[timing.flashScope||0];
    line.setValue('flashScope',value);

    line.refresh();
};






//=============================================================================
// EditorLine
//=============================================================================
function EditorLine(){
    this.initialize.apply(this, arguments);
}
EditorLine.prototype = Object.create(Sprite.prototype);
EditorLine.prototype.constructor = EditorLine;

EditorLine.TOP_MARGIN = 10;
EditorLine.RIGHT_MARGIN = 10;
EditorLine.MAX_WIDTH = 1024;
EditorLine.LINE_HEIGHT = 26;
EditorLine.FONT_SIZE = 16;
EditorLine.LINE_MARGIN = 4;
EditorLine.ELEM_MARGIN = 8;
EditorLine.prototype.initialize = function(index,selectionhandler=null) {
    Sprite.prototype.initialize.call(this);
    this.initMembers();
    this._index = index;
    this._selectionHandler = selectionhandler;

    this.x = Graphics.width - EditorLine.RIGHT_MARGIN;
    this.y = EditorLine.TOP_MARGIN+index*(EditorLine.LINE_HEIGHT+EditorLine.LINE_MARGIN);
    this.createContentsSprite();
};
EditorLine.prototype.initMembers = function(){
    this._index = -1;
    this._data = [];
    this._positions = [];
    this._selectedIndex = -1;
    this._active = false;

    this._contentsSprite = null;
    this._selectionSprite = null;
    this._elemMargin = EditorLine.ELEM_MARGIN;
    this._needsRefresh = false;

    this._selectionHandler = null;
};

EditorLine.prototype.destroy = function(){
    this._selectionHandler = null;
    this._data.length = 0;
    this._data = null;
    Sprite.prototype.destroy.apply(this,arguments);
};

EditorLine.ELEM_WIDTH = 64;
EditorLine.prototype.addElem = function(symbol,name,width=0,selectable=false,buttonHandler=null){
    var elem = this.createElem(this._data.length,symbol,name,width,selectable,buttonHandler);
    this._data.push(elem);
    this._needsRefresh = true;
};
EditorLine.prototype.createElem = function(idx,symbol,name,width=0,selectable=false,buttonHandler=null){
    return {
        idx:idx,
        symbol:symbol,
        value:'',
        name:name,
        width:width,
        selectable:selectable,
        buttonHandler:buttonHandler,
        focused:false,
        disabled:false
    }
};
EditorLine.prototype.setEnabled = function(symbol,noRefresh){
    this.setDisabled(symbol,noRefresh,false);
};
EditorLine.prototype.setDisabled = function(symbol,noRefresh=false,value=true){
    var elem = this.elem(symbol);
    if(!elem)return;
    if(elem.disabled!==value){
        elem.disabled = value;
        this._needsRefresh = true;
        if(!noRefresh){
            this.refresh();
        }
    }
};
EditorLine.prototype.setValue = function(symbol,value,noRefresh=false){
    var elem = this.elem(symbol);
    if(!elem)return;

    if(elem.value !== value){
        elem.value = value;
        this._needsRefresh = true;
        if(!noRefresh){
            this.refresh();
        }
    }
};
EditorLine.prototype.elem = function(symbol){
    var data = this._data;
    var length = data.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var elem = data[i];
        if(elem.symbol===symbol){
            return elem;
        }
    }
    return null;
}
            

EditorLine.prototype.setSelectionElem = function(index,num){
    var data = this._data;
    var length = data.length;
    for(var i=length-1; i>=0; i=(i-1)|0){
        var elem = data[i];
        if(elem.selectable){
            data.splice(i,1);
        }
    }

    var width = Editor.SELECTION_WIDHT;
    var selectable = true;
    var buttonHandler = null;
    for(var i=0; i<num;i=(i+1)|0){
        var idx = index+i;
        var elem = this.createElem(idx,i,i,width,selectable,buttonHandler);
        this._data.splice(idx,0,elem);
    }

    var length = data.length;
    for(var i=index+num; i<length; i=(i+1)|0){
        data[i].idx = i;
    }

    this._needsRefresh = true;
    this.refresh();
};

EditorLine.prototype.activate = function(){
    if(this._active===true)return;
    this._active = true;
    this.refreshSelectionSprite();
};
EditorLine.prototype.deactivate = function(){
    if(this._active===false)return;
    this._active = false;
    this.refreshSelectionSprite();
};

EditorLine.prototype.createContentsSprite = function(){
    var width = EditorLine.MAX_WIDTH;
    var height = EditorLine.LINE_HEIGHT;
    var bitmap = new Bitmap(width,height);
    bitmap.fontSize = EditorLine.FONT_SIZE;
    bitmap.outlineWidth = 4;
    bitmap.outlineColor = 'black';

    var sprite = new Sprite(bitmap);
    this.addChild(sprite);
    this._contentsSprite = sprite;
    sprite.anchor.x = 1;
};

EditorLine.prototype.setElemMargin = function(margin){
    this._elemMargin = margin;
};
EditorLine.prototype.elemMargin = function(){
    return this._elemMargin;
};

EditorLine.prototype.focusSymbol = function(symbol,noRefresh=false){
    var index = this.indexForSymbol(symbol);
    if(index>=0){
        this.focus(index,noRefresh);
    }else{
        this.unfocus(noRefresh);
    }
};
EditorLine.prototype.indexForSymbol = function(symbol){
    var data = this._data;
    var length = data.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var elem = data[i];
        if(elem && elem.symbol === symbol){
            return i;
        }
    }
    return -1;
}
EditorLine.prototype.focus = function(index,noRefresh=false){
    this.unfocus(true);
    this._data[index].focused = true;
    this._needsRefresh = true;
    if(!noRefresh){
        this.refresh();
    }
};

EditorLine.prototype.unfocus = function(noRefresh=false){
    for(const elem of this._data){
        if(elem.focused){
            this._needsRefresh = true;
        }
        elem.focused = false;
    }
    if(!noRefresh){
        this.refresh();
    }
};
EditorLine.prototype.refresh = function(){
    if(!this._needsRefresh)return;
    this._needsRefresh = false;
    this._positions.length = 0;

    var sprite = this._contentsSprite;
    var bitmap = sprite.bitmap;
    bitmap.clear();

    var data = this._data;
    var length = data.length;
    var x = 0;
    var elemMargin = this.elemMargin();

    for(var i = 0; i<length; i=(i+1)|0){
        var elem = data[i];
        if(elem.disabled)continue;

        if(i!==0){
            x = Math.ceil(x+elemMargin);
        }
        this._positions.push(x);
        x = this.drawElem(bitmap,x,elem);
        this._positions.push(x);
    }

    sprite.width = x;
};

EditorLine.prototype.drawElem = function(bitmap,x,elem){
    var text = elem.name;
    var width = elem.width;
    var height = bitmap.height;
    var align = elem.idx===0 ? align = 'right' :'center';

    bitmap.textColor = 'white';
    bitmap.outlineColor = 'black';
    bitmap.fontSize = EditorLine.FONT_SIZE;
    if(elem.value!==''){
        text += '：'+elem.value;
        align = 'left';
    }

    if(width===0){
        width = bitmap.measureTextWidth(text);
        width = Math.min(bitmap.width-x,width-4);
    }

    if(elem.buttonHandler){
        align = 'center';

        var color1,color2,textColor;
        if(elem.focused){
            color1 = 'rgba(255,150,50,0.5)';
            color2 = 'rgb(50,0,00)';
            textColor = 'rgb(255,255,220)';
        }else{
            color1 = 'rgba(50,150,255,0.5)';
            color2 = 'rgb(0,0,50)';
            textColor = 'rgb(220,255,255)';
        }
        bitmap.fontSize-=2;
        bitmap.fillRect(x,0,width,height,color1);
        strokeRect(bitmap,x+1,1,width-2,height-2,color2)
        bitmap.textColor = textColor;
    }

    bitmap.drawText(text,x+2,0,width-4,height,align);

    return x + width;
};

/* update
===================================*/
EditorLine.prototype.processTouch = function(){
    var y = TouchInput.y;
    var m = 4;
    if(y<this.y-m || y>this.y+EditorLine.LINE_HEIGHT+m)return false;

    var x = TouchInput.x;
    if(x>this.x+m)return false;

    var positions = this._positions;
    var length = positions.length;
    var baseX = this.x-this._contentsSprite.width;
    for(var i = 0; i<length; i=(i+2)|0){
        var idx = i/2;
        var x0 = baseX+positions[i];
        var x1 = baseX+positions[i+1];
        if(x<x0-m || x>x1+m)continue;

        var elem = this._data[idx];
        if(elem.buttonHandler){
            elem.buttonHandler();
        }
        if(elem.selectable){
            this._selectionHandler(elem.symbol);
        }
        return true;
    }
    return false;
};


/* select
===================================*/
EditorLine.prototype.select = function(index,force=false){
    if(!force && this._selectedIndex===index)return;
    this._selectedIndex = index;
    this.refreshSelectionSprite();
};
EditorLine.prototype.createSelectionSprite = function(){
    var width = 128;
    var height = EditorLine.LINE_HEIGHT;
    var bitmap = new Bitmap(width,height);
    var sprite = new Sprite(bitmap);
    this._selectionSprite = sprite;
    this.addChildAt(sprite,0);
};
EditorLine.prototype.refreshSelectionSprite = function(){
    var index = this._selectedIndex+1;
    if(index<0){
        if(this._selectionSprite){
            this._selectionSprite.visible = false;
        }
        return;
    }
    if(!this._selectionSprite){
        this.createSelectionSprite();
    }

    var sprite = this._selectionSprite;
    var bitmap = sprite.bitmap;
    bitmap.clear();

    var x0 = this._positions[index*2];
    var x1 = this._positions[index*2+1];
    sprite.x = -this._contentsSprite.width+x0;

    var color1,color2;
    if(this._active){
        color1 = 'rgba(255,150,0,0.5)';
        color2 = 'rgb(50,0,0)';
    }else{
        color1 = 'rgba(50,150,255,0.5)';
        color2 = 'rgb(0,0,50)';
    }

    var x = 0;
    var y = 0;
    var w = x1-x0;
    var h = bitmap.height;

    bitmap.fillRect(x,y,w,h,color1);
    strokeRect(bitmap,x+1,y+1,w-2,h-2,color2)
};


function strokeRect(bitmap,x,y,width,height,color){
    var context = bitmap.context;
    context.save();
    context.strokeStyle = color;
    context.strokeRect(x, y, width, height);
    context.restore();
    if(isMZ){
        bitmap._baseTexture.update();
    }else{
        bitmap._setDirty()
    }
}




//=============================================================================
// helper NameInput
//=============================================================================
function EditorNameInput(){
    this.initialize.apply(this, arguments);
}
EditorNameInput.prototype.initialize = function(def,handler){
    this._name = '';
    this._default = def;
    this._changeHandler = handler;
};
EditorNameInput.prototype.releaseHandler = function(){
    this._changeHandler = null;
};
EditorNameInput.prototype.name = function(){
    return this._name;
};
EditorNameInput.prototype.refresh = function(){
    this._changeHandler(this._name);
};
EditorNameInput.prototype.restoreDefault = function(){
    this._name = this._default;
    this.refresh();
    return true;
};
EditorNameInput.prototype.add = function(ch){
    this._name += ch;
    this.refresh();
    return true;
};
EditorNameInput.prototype.back = function(){
    if(this._name.length>=1){
        this._name = this._name.substring(0,this._name.length-1);
        this.refresh();
        return true;
    }else{
        return false;
    }
};

//=============================================================================
// PickerBase
//=============================================================================
function PickerBase(){
    this.initialize.apply(this, arguments);
};
PickerBase.TINT_SEVERAL = 0xaaffff;
PickerBase.TINT_NORMAL = 0xffffaa;
PickerBase.LAYOUT = {
    marginTopBottom:5
};

PickerBase.prototype = Object.create(PIXI.Container.prototype);
PickerBase.prototype.constructor = PickerBase;
PickerBase.prototype.initialize = function() {
    PIXI.Container.call(this);
    this.initMembers();

    this.createBackSprite();
    this.createHighlightBitmap();
    this.createGuideSprite();
    this.createHeaderSprite();
};

PickerBase.prototype.initMembers = function(){
    this._header = '';
    this._headerSprite = null;

    this._topRow = 0;
    this._maxRow = 0;
    this._dispRows = 0;
    this._maxTopRow = 0;

    this._owner = null;
    this._severalMode = false;
    this._severalModeSwitched = false;
    this._selectingIndexes = [];

    this._backSprite = null;
    this._highlightSprites = [];
    this._highlightBitmap = null;
    this._guideSprite = null;

    this._listType = null;

    this._categoryIndex = 0;
};

PickerBase.prototype.startPicking = function(owner){
    this.visible = true;
    this._owner = owner;

    Input.clear();
    TouchInput.clear();
    this.registerWheelListener();

    this.refresh();
    this._headerSprite.opacity = 255;
    this._headerSprite.visible = true;

    if(this._guideSprite){
        var sprite = this._guideSprite;
        sprite.x = this._width + 10;
        sprite.y = Graphics.boxHeight-10;
    }
};

PickerBase.prototype.end = function(){
    if(this._owner){
        this._owner.didEndPicking();
    }
    this._owner = null;

    this.resignWheelListener();
    this.visible = false;


    SoundManager.playCancel();
    Input.clear();
    TouchInput.clear();
};

PickerBase.prototype.refresh = function(){
    var type = this.categoryType();
    if(this._listType === type)return;
    this.setListType(type);

    if(this.isReady()){
        this._refresh();
    }
};

PickerBase.prototype.isReady = function(){
    return true;
};

PickerBase.prototype._refresh = function(){
    var col = this.maxColumns();

    var itemWidth = this.itemWidth();
    var itemHeight = this.itemHeight();

    var margin = PickerBase.LAYOUT.marginTopBottom;
    var itemNum = this.maxItems();

    var mx = this.itemMarginX();
    var my = this.itemMarginY();

    this._maxRow = Math.ceil(itemNum/col);
    this._dispRows = Math.floor((Graphics.boxHeight-2*margin)/(itemHeight+my));
    this._maxTopRow = Math.max(0,this._maxRow-this._dispRows-1);

    var row = this._maxRow;
    var width = itemWidth*col+mx*(col-1)+margin*2;
    this._width = width;
    this._height = Graphics.boxHeight;

    this.refreshBackSprite();
    this.refreshItems();
};
PickerBase.prototype.setListType = function(type){
    this._listType = type;
    this.refreshHeaderSprite();
};

PickerBase.prototype.refreshItems = function(){};


/* needs overwrite
===================================*/
PickerBase.prototype.maxColumns = function(){return 4;};
PickerBase.prototype.itemHeight = function(){return 48;};
PickerBase.prototype.itemWidth = function(){return 48;};
PickerBase.prototype.maxItems = function(){return 0;};
PickerBase.prototype.guideTexts = function(){return null;};
PickerBase.prototype.itemMarginX = function(){return 0;};
PickerBase.prototype.itemMarginY = function(){return 0;};
PickerBase.prototype.categoryType = function(){
    return 1;
};
PickerBase.prototype.headerText = function(){return '';}

//category
PickerBase.prototype.maxCategories = function(){return 1};
PickerBase.prototype.isCategoryValid = function(index){return true};
PickerBase.prototype.isSeveralModeValid = function(){return true};
PickerBase.prototype.applyData = function(){};


/* select
===================================*/
PickerBase.prototype.deselectIndex = function(index){
    if(index<0)return;
    var arrayIdx = this._selectingIndexes.indexOf(index);
    if(arrayIdx<0)return;

    this._selectingIndexes.splice(arrayIdx,1);
    var sprite = this._highlightSprites[arrayIdx];
    if(sprite){
        this._highlightSprites.splice(arrayIdx,1);
        sprite.parent.removeChild(sprite);
    }
};

PickerBase.prototype.didPickData = function(index){
    if(this._selectingIndexes.contains(index)){
        if(this.canDeselect()){
            this.deselectIndex(index);
        }
    }else{
        this.setSelectingIndex(index);
    }
    this.applyData();
};
PickerBase.prototype.canDeselect = function(){
    return true;
};




/* update
===================================*/
PickerBase.prototype.update = function(){
    if(this._headerSprite.visible){
        this.updateHeaderSprite();
    }

    if(Input._latestButton){
        this.processInput();
    }else if(TouchInput.isLongPressed() && this.isSeveralModeValid()){
        if(!this._severalModeSwitched){
            this.switchSelectingMode();
            this._severalModeSwitched = true;
            if(this._selectingIndexes.length===0){
                this.processTouch();
            }
            this.applyData();
        }
    }else if(TouchInput.isTriggered()){
        this.processTouch();
        this._severalModeSwitched = false;
    }
};
PickerBase.prototype.onKeyDown = function(event){
    var keyCode = event.keyCode;
    if(keyCode>=KEY_CODE.alphabet && keyCode<=KEY_CODE.alphabetEnd){
        var chara = event.key;
        this.search(chara);
    }
};
PickerBase.prototype.search = function(chara){};

PickerBase.prototype.processTouch = function(){
    var x = TouchInput.x - this.x;
    var y = TouchInput.y - this.y;

    if(x<0 || x>this._width){
        this.end();
        return;
    }

    var maxCol = this.maxColumns();
    var length = this.maxItems();
    var margin = PickerBase.LAYOUT.marginTopBottom;
    var mx = this.itemMarginX();
    var my = this.itemMarginY();
    var itemWidth = this.itemWidth();
    var itemHeight = this.itemHeight();
    var colW = itemWidth+mx;
    var rowH = itemHeight+my;
    var x0 = margin/2;

    var ix = x0;
    var iy = margin/2;
    for(var i = 0; i<length; i=(i+1)|0){
        if(i===0){
        }else if(i%maxCol === 0){
            ix = margin/2;
            iy += rowH;
        }else{
            ix += colW;
        }
        if(ix<=x && x<=ix+colW && iy<=y && y<=iy+rowH){
            this.didPickData(i);
            return;
        }
    }
};

PickerBase.prototype.setSelectingIndex = function(index,enableOverlap=false){
    SoundManager.playCursor();
    var sprite = null;
    var noSelect = index<0;

    if(this._severalMode){
        if(!enableOverlap&&this._selectingIndexes.contains(index)){
            return;
        }
        this._selectingIndexes.push(index);
    }else{
        this._selectingIndexes[0] = index;
        sprite = this._highlightSprites[0];
        if(noSelect){
            if(sprite){
                this._highlightSprites.length = 0;
                sprite.parent.removeChild(sprite);
            }
            return;
        }
    }
    if(noSelect)return;

    if(!sprite){
        sprite = this.pushNewHighlightSprite();
    }
    sprite.tint = this._severalMode?PickerBase.TINT_SEVERAL:PickerBase.TINT_NORMAL;
    

    var maxCol = this.maxColumns();
    var margin = PickerBase.LAYOUT.marginTopBottom;
    var mx = this.itemMarginX();
    var my = this.itemMarginY();
    var itemWidth = this.itemWidth();
    var itemHeight = this.itemHeight();

    var col = index%maxCol;
    var row = Math.floor(index/maxCol);
    sprite.visible = true;
    sprite.x = margin + col*itemWidth+(col-1)*mx;
    sprite.y = margin + row*itemHeight+(row-1)*my;
};

PickerBase.prototype.deselectAll = function(){
    var sprites = this._highlightSprites;
    var length = sprites.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var sprite = sprites[i];
        sprite.parent.removeChild(sprite);
    }
    sprites.length = 0;
    this._selectingIndexes.length = 0;
};

PickerBase.prototype.selectAll = function(){
    var length = this.maxItems();
    for(var i=0; i<length; i=(i+1)|0){
        this.setSelectingIndex(i);
    }
};




/* headerSprite
===================================*/
PickerBase.prototype.createHeaderSprite = function(){
    var bitmap = new Bitmap(256,24);
    var sprite = new Sprite(bitmap);
    bitmap.fontSize = 21;
    bitmap.textColor = 'white'
    bitmap.outlineWidth = 6;
    bitmap.outlineColor = 'rgb(0,0,200)';
    this.addChild(sprite);
    this._headerSprite = sprite;
};
PickerBase.prototype.refreshHeaderSprite = function(){
    var header = this.headerText();
    if(header === this._header)return;
    this._header = header;
    var sprite = this._headerSprite;
    var bitmap = sprite.bitmap;
    bitmap.clear();
    bitmap.drawText(header,1,0,bitmap.width-2,bitmap.height);
    sprite.opacity = 255;
    sprite.visible = true;

    this.addChild(sprite);
};
PickerBase.prototype.updateHeaderSprite = function(){
    if(this._headerSprite.opacity>200){
        this._headerSprite.opacity -= 1;
    }else{
        this._headerSprite.opacity -= 5;
    }
    if(this._headerSprite.opacity<=0){
        this._headerSprite.visible = false;
    }
};


/* backSprite
===================================*/
PickerBase.prototype.createBackSprite = function(){
    var sprite,bitmap;
    bitmap = new Bitmap(16,16);
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    this._backSprite = sprite;
    sprite.opacity = 150;
    bitmap.fillAll('black');
};
PickerBase.prototype.refreshBackSprite = function(){
    var width = this._width;
    var height = Graphics.boxHeight;
    var sprite = this._backSprite;
    sprite.scale.set(width/16,height/16);
};


/* highlight sprites
===================================*/
PickerBase.prototype.createHighlightBitmap = function(){
    var bitmap = new Bitmap(16,16);
    bitmap.fillAll('white');
    this._highlightBitmap = bitmap;
};
PickerBase.prototype.pushNewHighlightSprite = function(){
    var itemHeight = this.itemHeight();
    var itemWidth = this._width / this.maxColumns();

    var bitmap = this._highlightBitmap;
    var sprite = new Sprite(bitmap);
    this._highlightSprites.push(sprite);
    this.addChild(sprite);
    sprite.opacity = 100;
    sprite.scale.set(itemWidth/bitmap.width,itemHeight/bitmap.height);
    return sprite;
};

/* guide sprite
===================================*/
PickerBase.prototype.createGuideSprite = function(){
    var texts = this.guideTexts();
    if(!texts)return;

    var fontSize = 14;
    var width = 200;
    var lineHeight = fontSize + 4;
    var lines = texts.length;
    var height = lineHeight*lines;
    var bitmap = new Bitmap(width,height);
    var sprite = new Sprite(bitmap);
    this.addChild(sprite);
    this._guideSprite = sprite;

    sprite.anchor.set(0,1);

    bitmap.fontSize =  fontSize;
    bitmap.fillAll('rgb(0,0,150,0.6)');

    var y = 0;
    var length = texts.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var text = texts[i];
        bitmap.drawText(text,1,y,width-2,lineHeight);
        y += lineHeight;
    };
};

/* scroll
===================================*/
PickerBase.prototype.setTopIndex = function(i){
    var row = Math.floor(i/this.maxColumns());
    var newRow = row.clamp(0,this._maxTopRow);
    if(this._topRow === newRow)return;
    this._topRow = newRow;
    this.refreshPosition();
};
PickerBase.prototype.setTopRowNext = function(){
    var index = (this._topRow-1)*this.maxColumns()
    this.setTopIndex(index);
};
PickerBase.prototype.setTopRowPrevious = function(){
    var index = (this._topRow+1)*this.maxColumns();
    this.setTopIndex(index);
};
PickerBase.prototype.refreshPosition = function(){
    var oldY = this.y
    this.y = -this._topRow*this.itemHeight();
    if(this._topRow!==0){
        this.y -= PickerBase.LAYOUT.marginTopBottom;
    }

    var dy = this.y - oldY;
    this._backSprite.y -= dy;
    if(this._guideSprite){
        this._guideSprite.y -= dy;
    }
};



PickerBase.prototype.processInput = function(){
    if(Input.isTriggered('ok')||Input.isTriggered('cancel')){
        this.end();
    }else if(Input.isRepeated('up')){
        SoundManager.playCursor();
        this.setTopRowNext();
    }else if(Input.isRepeated('down')){
        SoundManager.playCursor();
        this.setTopRowPrevious();
    }else if(Input.isRepeated('left')){
        this.processPageDown();
    }else if(Input.isRepeated('right')){
        this.processPageUp();
    }
};
PickerBase.prototype.processPageUp = function(){
    SoundManager.playCursor();
    if(this.maxCategories()>1){
        this._topRow = 0;
        var index = this._categoryIndex;
        do{
            index += 1;
            if(index>=this.maxCategories()){
                index = 0;
            }
        }while(index!==this._categoryIndex && !this.isCategoryValid(index));
        this._categoryIndex = index;
        this.refreshCategory();
    }else{
        this._topRow = Math.max(0,this._topRow-this._dispRows);
        this.refreshPosition();
    }
};
PickerBase.prototype.processPageDown = function(){
    SoundManager.playCursor();
    if(this.maxCategories()>1){
        this._topRow = 0;
        var index = this._categoryIndex;
        do{
            index -= 1;
            if(index<0){
                index = this.maxCategories()-1;
            }
        }while(index!==this._categoryIndex && !this.isCategoryValid(index));
        this._categoryIndex = index;
        this.refreshCategory();
    }else{
        this._topRow = Math.min(this._maxTopRow,this._topRow+this._dispRows);
        this.refreshPosition();
    }
};

PickerBase.prototype.refreshCategory = function(){
    this.deselectAll();
    this.refresh();
    this.refreshPosition();
};



/* several mode
===================================*/
PickerBase.prototype.switchSelectingMode = function(){
    this.setSeveralMode(!this._severalMode);
};
PickerBase.prototype.setSeveralMode = function(valid){
    if(this._severalMode===valid)return;
    this._severalMode = valid;

    this.deselectAll();
};



/* wheel
===================================*/
PickerBase.prototype.registerWheelListener = function(){
    var listener = this._onWheel.bind(this);
    this._wheelListener = listener;
    document.addEventListener('wheel', listener);
};
PickerBase.prototype.resignWheelListener = function(){
    if(!this._wheelListener)return;

    document.removeEventListener('wheel', this._wheelListener);
    this._wheelListener = null;
};

PickerBase.prototype._onWheel = function(event) {
    if(event.deltaY>0){
        this.setTopRowNext();
    }else if(event.deltaY<0){
        this.setTopRowPrevious();
    }
    event.stopPropagation();
};


//=============================================================================
// ImagePickerBase
//=============================================================================
function ImagePickerBase(){
    this.initialize.apply(this, arguments);
};
ImagePickerBase.prototype = Object.create(PickerBase.prototype);
ImagePickerBase.prototype.constructor = ImagePickerBase;
ImagePickerBase.prototype.initialize = function() {
    PickerBase.prototype.initialize.call(this);
    this._keydownListener = null;
};
ImagePickerBase.prototype.applyData = function(){
    if(this._owner){
        var imageName = this.imageName();
        this._owner.setImage(imageName);
    }
};
ImagePickerBase.prototype.imageName = function(){return '';};

ImagePickerBase.prototype.end = function(){
    PickerBase.prototype.end.call(this);
    if(this._keydownListener){
        document.removeEventListener('keydown',this._keydownListener);
        this._keydownListener = null;
    }
};

ImagePickerBase.prototype.startPicking = function(owner){
    var listener = this.onKeyDown.bind(this);
    this._keydownListener = listener;
    document.addEventListener('keydown', listener);

    PickerBase.prototype.startPicking.call(this,owner);
};

ImagePickerBase.prototype.search = function(chara){
    var imageNames = this._imageNames;
    var length = imageNames.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var image = imageNames[i];
        if(image[0].toLowerCase()===chara){
            SoundManager.playCursor();
            if(this._categoryIndex===i){
                this._headerSprite.opacity = 255;
                this._headerSprite.visible = true;
            }else{
                this._categoryIndex = i;
                this._topRow = 0;
                this.refreshCategory();
            }
            return;
        }
    }
};




//=============================================================================
// ImagePicker
//=============================================================================
function ImagePicker(){
    this.initialize.apply(this, arguments);
};

ImagePicker.MAX_COL = 5;
ImagePicker.IMAGE_SCALE = 1/3;

ImagePicker.IMAGE_PATH = 'img/animations/';
ImagePicker._imageNames = null;
ImagePicker.imageNames = function(){
    if(this._imageNames){
        return this._imageNames;
    }

    var fs = require('fs');
    var path = require('path');
    var base = path.dirname(process.mainModule.filename);
    var dirPath = path.join(base,this.IMAGE_PATH);

    var animationNames = [];
    var files = fs.readdirSync(dirPath)
    var length = files.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var file = files[i];
        var fp = path.join(dirPath,file);

        if(!fs.statSync(fp).isFile())continue;
        if(!(/.*\.png$/.test(file)))continue;
        animationNames.push(file.replace('.png',''));
    }
    this._imageNames = animationNames.sort();
    return this._imageNames;
};

ImagePicker.prototype = Object.create(ImagePickerBase.prototype);
ImagePicker.prototype.constructor = ImagePicker;
ImagePicker.prototype.initialize = function(){
    ImagePickerBase.prototype.initialize.call(this);

    this._imageNames = ImagePicker.imageNames();
    this._imageData = null;

    this._maxItems = 0;
    this._imageSize = this.imageSize();
    this._imageRows = 0;
    this._imageCols = 0;


    this._bitmap = null;

    var scale = this.imageScale();
    var sprite = new Sprite();
    this._imageSprite = sprite;
    this.addChild(sprite);
    sprite.scale.set(scale,scale);
};


/* overwrite setting
===================================*/
ImagePicker.prototype.maxItems = function(){
    return this._maxItems;
};
ImagePicker.prototype.maxColumns = function(){
    return ImagePicker.MAX_COL;
};
ImagePicker.prototype.itemHeight = function(){
    return this.imageSize()*this.imageScale();
};
ImagePicker.prototype.itemWidth = function(){
    return this.imageSize()*this.imageScale();
};
ImagePicker.prototype.itemMarginX = function(){
    return 0;
};
ImagePicker.prototype.itemMarginY = function(){
    return 0;
};
ImagePicker.prototype.guideTexts = function(){
    return [
        '１つのアニメーション画像のみから選択可',
        '←→キー、画像切り替え',
        '決定キー、リスト外クリックで選択終了',
        'ctrl+R：プレビュー切り替え',
        null,
        'ctrl+M：画像の反転',
    ];
};
ImagePicker.prototype.imageName = function(){
    var imageName = this._imageNames[this._categoryIndex];
    var name = ''

    var indexes = this._selectingIndexes;
    var length = indexes.length;
    for(var i = 0; i<length; i=(i+1)|0){
        if(i>0){
            name += ',';
        }
        var index = indexes[i];
        name +='ANIM:'+imageName+':'+index;
    }
    return name;
};
ImagePicker.prototype.headerText = function(){
    var imageName = this._imageNames[this._categoryIndex];
    var text = '【'+imageName+'】('+(this._categoryIndex+1)+'/'+this._imageNames.length+')';
    return text;
};
ImagePicker.prototype.categoryType = function(){
    return this._categoryIndex;
};
ImagePicker.prototype.maxCategories = function(){
    return this._imageNames ? this._imageNames.length : 1; 
};
ImagePicker.prototype.isCategoryValid = function(index){
    return !!this._imageNames[index];
};

ImagePicker.prototype.processInput = function(){
    if(Input.isPressed('control')&&Input.isTriggered(motionKey)){
        //mirror
        this._owner.setImageMirror();
        SoundManager.playCursor();
    }else{
        ImagePickerBase.prototype.processInput.call(this);
    }
}

/* originalSetting
===================================*/
ImagePicker.prototype.imageSize = function(){
    var name = this._imageNames[this._categoryIndex];
    var sizeMatch = name.match(/_x([0-9]+)$/);
    if(sizeMatch){
        return Number(sizeMatch[1]);
    }else{
        return 192;
    }
};
ImagePicker.prototype.imageScale = function(){
    return ImagePicker.IMAGE_SCALE;
};

/* refresh
===================================*/
ImagePicker.prototype.setListType = function(type){
    ImagePickerBase.prototype.setListType.call(this,type);
            
    this._imageCols = 0;
    this._imageRows = 0;
    this._maxItems = 0;

    var bitmap = this.bitmap();
    this._bitmap = bitmap;
    bitmap.addLoadListener(ImagePicker.prototype._setListType.bind(this));
};
ImagePicker.prototype._setListType = function(bitmap){
    if(this._bitmap !== bitmap)return;

    var imageSize = this.imageSize();
    var width = bitmap.width;
    var height = bitmap.height;
    var imageCols = Math.floor(width/imageSize);
    var imageRows = Math.floor(height/imageSize);
    this._imageCols = imageCols;
    this._imageRows = imageRows;

    var maxCol = this.maxColumns();
    var validCols = Math.ceil(imageCols/maxCol)*maxCol;
    this._maxItems = validCols*imageRows;

    this._refresh();
};

ImagePicker.prototype.refreshItems = function(){
    var imageSize = this.imageSize();

    var bitmap = this.bitmap();
    var width = bitmap.width;
    var height = bitmap.height;
    var imageCols = Math.floor(width/imageSize);
    var imageRows = Math.floor(height/imageSize);
    var maxCol = this.maxColumns();
    var margin = PickerBase.LAYOUT.marginTopBottom;

    var sprite = this._imageSprite;
    sprite.bitmap = bitmap;
    sprite._frame.width = maxCol*imageSize;
    sprite._refresh();
    sprite.x = margin;
    sprite.y = margin;
};

ImagePicker.prototype.bitmap = function(){
    var imageName = this._imageNames[this._categoryIndex];
    return ImageManager.loadAnimation(imageName);
};


/* start picking
===================================*/
ImagePicker.prototype.startPicking = function(owner,imagestr,imageData){
    var frames = [];
    this._imageData = imageData;
    if(imagestr){
        var images = imagestr.split(',');

        var length = images.length;
        var index = -1;
        for(var i = 0; i<length; i=(i+1)|0){
            var image = images[i];
            if(image.indexOf('ANIM:')===0){
                var elems = image.split(':');
                var animImage = elems[1];
                if(index<0){
                    index = this._imageNames.indexOf(animImage);
                }
                var frame = Number(elems[2]);
                frames.push(frame)
            }
        }
        this._categoryIndex = index;
    };
    if(this._categoryIndex<0){
        this._categoryIndex = 0;
    }

    ImagePickerBase.prototype.startPicking.call(this,owner);
    
    var rows = this._imageRows;
    var cols = this.maxColumns();
    var length = frames.length;

    this.setSeveralMode(false);
    for(var i = 0; i<length; i=(i+1)|0){
        var frame = frames[i];

        var col,row;
        var col = frame%cols;
        var row = Math.floor(frame/cols);
        if(col >= cols){
            row += rows;
            col -= cols;
        }
        var index = row*cols + col;
        this.setSelectingIndex(index);
    }
};








//=============================================================================
// AnimationPicker
//=============================================================================
function AnimationPicker(){
    this.initialize.apply(this, arguments);
};

AnimationPicker.prototype = Object.create(ImagePicker.prototype);
AnimationPicker.prototype.constructor = AnimationPicker;
AnimationPicker.prototype.initialize = function(){
    ImagePicker.prototype.initialize.call(this);

    this._loop = false;
    this._speed = 4;
};


/* overwrite setting
===================================*/
AnimationPicker.prototype.guideTexts = function(){
    return [
        '１つのアニメーション画像のみから選択可',
        '←→キー、画像切り替え',
        '決定キー、リスト外クリックで選択終了',
        null,
        'アニメーション順にセルを選択',
        'ctrl+M：画像の反転',
        'Backspace：選択をクリア',
        '数字キー：アニメスピードの変更',
        'ctrl+L：ループON/OFFの変更',
        '(ループを止めるには次のモーションでアニメ設定)'
    ];
};



/* input
===================================*/
AnimationPicker.prototype.processInput = function(){
    if(Input.isPressed('control')&&Input.isTriggered(lockKey)){
        this.switchLoopValue();
    }else if(Input.isPressed('control')&&Input.isTriggered(motionKey)){
        //mirror
        this._owner.setAnimationMirror();
        SoundManager.playCursor();
    }else if(Input.isTriggered(backspaceKey)){
        if(this._selectingIndexes.length>0){
            this.deselectAll();
            SoundManager.playCancel();
        }else{
            this.selectAll();
            SoundManager.playCursor();
        }
        this.applyData();
    }else{
        ImagePicker.prototype.processInput.call(this);
    }
};

AnimationPicker.prototype.onKeyDown = function(event){
    var keyCode = event.keyCode;

    var numKeyCode = KEY_CODE.num;
    var tenKeyCode = KEY_CODE.tenkey;
    var num = null;
    if(keyCode>=numKeyCode&&keyCode<numKeyCode+10){
        num = Number(keyCode-numKeyCode);
    }else if(keyCode>=tenKeyCode&&keyCode<tenKeyCode+10){
        num = Number(keyCode-tenKeyCode);
    }

    if(num!==null){
        this.changeSpeed(num);
    }else{
        ImagePickerBase.prototype.onKeyDown.call(this,event);
    }
};


AnimationPicker.prototype.switchLoopValue = function(){
    this._loop = !this._loop;
    SoundManager.playCursor();

    if(this._selectingIndexes.length>0){
        this.applyData();
    }
};
AnimationPicker.prototype.changeSpeed = function(speed){
    if(this._speed === speed)return;
    this._speed = speed;
    SoundManager.playCursor();

    if(this._selectingIndexes.length>0){
        this.applyData();
    }
};


/* start picking
===================================*/
AnimationPicker.prototype.startPicking = function(owner,imageData){
    var frames = [];
    if(imageData){
        this._speed = imageData.speed;
        this._loop = imageData.loop;    
        var images = imageData.images;

        var length = images.length;
        var index = -1;
        for(var i = 0; i<length; i=(i+1)|0){
            var image = images[i];
            if(image.indexOf('ANIM:')===0){
                var elems = image.split(':');
                var animImage = elems[1];
                if(index<0){
                    index = this._imageNames.indexOf(animImage);
                }
                var frame = Number(elems[2]);
                frames.push(frame)
            }
        }
        this._categoryIndex = index;
    };
    if(this._categoryIndex<0){
        this._categoryIndex = 0;
    }

    ImagePickerBase.prototype.startPicking.call(this,owner);
    
    var rows = this._imageRows;
    var cols = this.maxColumns();
    var length = frames.length;

    this.setSeveralMode(true);
    for(var i = 0; i<length; i=(i+1)|0){
        var frame = frames[i];

        var col,row;
        var col = frame%cols;
        var row = Math.floor(frame/cols);
        if(col >= cols){
            row += rows;
            col -= cols;
        }
        var index = row*cols + col;
        this.setSelectingIndex(index);
    }
};

AnimationPicker.prototype.applyData = function(){
    if(!this._owner)return;

    if(this._selectingIndexes.length===0){
        this._owner.setAnimation(null);
        return;
    }

    var images = this.imageNames();
    this._owner.setAnimation(images,this._speed,this._loop);
};

AnimationPicker.prototype.imageNames = function(){
    var imageName = this._imageNames[this._categoryIndex];
    var names = [];

    var indexes = this._selectingIndexes;
    var length = indexes.length;

    for(var i = 0; i<length; i=(i+1)|0){
        var index = indexes[i];
        names.push('ANIM:'+imageName+':'+index);
    }
    return names;
}; 


AnimationPicker.prototype.didPickData = function(index){
    this.setSelectingIndex(index,true);
    this.applyData();
};



//=============================================================================
// EasingPicker
//=============================================================================
function EasingPicker(){
    this.initialize.apply(this, arguments);
};

EasingPicker.MARGIN = 5;
EasingPicker.ROW_HEIGHT = 20;
EasingPicker.COL_WIDTH = 300;
EasingPicker.MAX_COL = 1;

EasingPicker.LIST = [
    'linear',
    'easeIn','easeOut','easeInOut',
    'quadIn','quadOut','quadInOut',
    'cubicIn','cubicOut','cubicInOut',
    'quartIn','quartOut','quartInOut',
    'quintIn','quintOut','quintInOut',
    'sineIn','sineOut','sineInOut',
    'expoIn','expoOut','expoInOut',
    'circIn','circOut','circInOut',
    'bounceOut',
    'backIn','backOut','backInOut',
    'elastic',
    'swingFromTo','swingFrom','swingTo',
    'bounce','bouncePast',
    'easeFromTo','easeFrom','easeTo'
];

EasingPicker.prototype = Object.create(PickerBase.prototype);
EasingPicker.prototype.constructor = ImagePicker;
EasingPicker.prototype.initialize = function(){
    PickerBase.prototype.initialize.call(this);

    this._list = EasingPicker.LIST;
    this._contentsSprite = null;
    this.createContentsSprite();
};
EasingPicker.prototype.createContentsSprite = function(){
    var length = this.maxItems();

    var width = this.itemWidth();
    var height = this.itemHeight()*length;

    var bitmap = new Bitmap(width,height);
    var sprite = new Sprite(bitmap);
    this.addChild(sprite);
    this._contentsSprite = sprite;
    sprite.y = PickerBase.LAYOUT.marginTopBottom;
};

/* overwrite setting
===================================*/
EasingPicker.prototype.maxItems = function(){
    return this._list.length;
};
EasingPicker.prototype.maxColumns = function(){
    return EasingPicker.MAX_COL;
};
EasingPicker.prototype.itemHeight = function(){
    return EasingPicker.ROW_HEIGHT;
};
EasingPicker.prototype.itemWidth = function(){
    return EasingPicker.COL_WIDTH;
};
EasingPicker.prototype.itemMarginX = function(){
    return EasingPicker.MARGIN;
};
EasingPicker.prototype.itemMarginY = function(){
    return 0;
};
EasingPicker.prototype.guideTexts = function(){
    return [
        'イージング（動作の緩急）を選択',
        '↑↓キー、マウスホイールでスクロール',
        '決定キー、リスト外クリックで選択終了',
        'ctrl+R：プレビューON/OFF',
    ];
};
EasingPicker.prototype.headerText = function(){
    return '【イージングの選択】';
};
EasingPicker.prototype.isCategoryValid = function(){
    return false;
};
EasingPicker.prototype.categoryType = function(){
    return 0;
};
EasingPicker.prototype.maxCategories = function(){
    return 1;
};
EasingPicker.prototype.isSeveralModeValid = function(){return false};
EasingPicker.prototype.applyData = function(){
    if(!this._owner)return;

    var easing = this._list[this._selectingIndexes[0]];
    this._owner.setEasing(easing);
};

EasingPicker.prototype.canDeselect = function(){
    return false;
};

/* refresh
===================================*/
EasingPicker.prototype.refreshItems = function(){
    var width = this.itemWidth();
    var lineHeight = this.itemHeight();
    var list = this._list;
    var length = list.length;
    var height = lineHeight * length;

    var bitmap = this._contentsSprite.bitmap;
    bitmap.clear();

    var margin = 5;
    var names = this._names;
    var length = list.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var text = list[i];
        var x = margin;
        var y = i*lineHeight;

        bitmap.fontSize = lineHeight-3;
        bitmap.drawText(text,x,y,width,lineHeight);
    }
};

/* start picking
===================================*/
EasingPicker.prototype.startPicking = function(owner,easing){
    this._categoryIndex = 0;

    PickerBase.prototype.startPicking.call(this,owner);

    this.deselectAll();
    var index = this._list.indexOf(easing);
    if(index>=0){
        this.setSelectingIndex(index);
    }
};







//=============================================================================
// SePicker
//=============================================================================
function SePicker(){
    this.initialize.apply(this, arguments);
};

SePicker.MARGIN = 5;
SePicker.ROW_HEIGHT = 20;
SePicker.COL_WIDTH = 300;
SePicker.MAX_COL = 1;
SePicker.SE_PATH = 'audio/se/';
SePicker._seNames = null;
SePicker.seNames = function(){
    if(this._seNames){
        return this._seNames;
    }

    var fs = require('fs');
    var path = require('path');
    var base = path.dirname(process.mainModule.filename);
    var dirPath = path.join(base,this.SE_PATH);

    var seNames = [];
    var files = fs.readdirSync(dirPath)
    var length = files.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var file = files[i];
        var fp = path.join(dirPath,file);

        if(!fs.statSync(fp).isFile())continue;
        if(!(/.*\.ogg$/.test(file)))continue;
        seNames.push(file.replace('.png',''));
    }
    this._seNames = seNames.sort();
    this._seNames.unshift('なし');
    return this._seNames;
};


SePicker.prototype = Object.create(PickerBase.prototype);
SePicker.prototype.constructor = ImagePicker;
SePicker.prototype.initialize = function(){
    PickerBase.prototype.initialize.call(this);

    this._keydownListener = null;
    this._list = SePicker.seNames();
    this._contentsSprite = null;
    this.createContentsSprite();
};
SePicker.prototype.end = function(){
    PickerBase.prototype.end.call(this);
    if(this._keydownListener){
        document.removeEventListener('keydown',this._keydownListener);
        this._keydownListener = null;
    }
};
SePicker.prototype.createContentsSprite = function(){
    var length = this.maxItems();

    var width = this.itemWidth();
    var height = this.itemHeight()*length;

    var bitmap = new Bitmap(width,height);
    var sprite = new Sprite(bitmap);
    this.addChild(sprite);
    this._contentsSprite = sprite;
    sprite.y = PickerBase.LAYOUT.marginTopBottom;
};

/* overwrite setting
===================================*/
SePicker.prototype.maxItems = function(){
    return this._list.length;
};
SePicker.prototype.maxColumns = function(){
    return SePicker.MAX_COL;
};
SePicker.prototype.itemHeight = function(){
    return SePicker.ROW_HEIGHT;
};
SePicker.prototype.itemWidth = function(){
    return SePicker.COL_WIDTH;
};
SePicker.prototype.itemMarginX = function(){
    return SePicker.MARGIN;
};
SePicker.prototype.itemMarginY = function(){
    return 0;
};
SePicker.prototype.guideTexts = function(){
    return [
        'SEファイルを選択',
        '↑↓キー、マウスホイールでスクロール',
        '決定キー、リスト外クリックで選択終了'
    ];
};
SePicker.prototype.headerText = function(){
    return '【SEファイルの選択】';
};
SePicker.prototype.isCategoryValid = function(){
    return false;
};
SePicker.prototype.categoryType = function(){
    return 0;
};
SePicker.prototype.maxCategories = function(){
    return 1;
};
SePicker.prototype.isSeveralModeValid = function(){return false};
SePicker.prototype.applyData = function(){
    if(!this._owner)return;

    var seName = this._list[this._selectingIndexes[0]];
    if(seName === 'なし'){
        seName = '';
    }else{
        seName = seName.replace('.ogg','');
    }
    this._owner.setSeName(seName);
};

/* refresh
===================================*/
SePicker.prototype.refreshItems = function(){
    var width = this.itemWidth();
    var lineHeight = this.itemHeight();
    var list = this._list;
    var length = list.length;
    var height = lineHeight * length;

    var bitmap = this._contentsSprite.bitmap;
    bitmap.clear();

    var margin = 5;
    var names = this._names;
    var length = list.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var text = list[i];
        var x = margin;
        var y = i*lineHeight;

        bitmap.fontSize = lineHeight-3;
        bitmap.drawText(text,x,y,width,lineHeight);
    }
};

/* start picking
===================================*/
SePicker.prototype.startPicking = function(owner,easing){
    var listener = this.onKeyDown.bind(this);
    this._keydownListener = listener;
    document.addEventListener('keydown', listener);

    this._categoryIndex = 0;

    PickerBase.prototype.startPicking.call(this,owner);

    this.deselectAll();
    var index = this._list.indexOf(easing);
    if(index>=0){
        this.setSelectingIndex(index);
    }
};
SePicker.prototype.search = function(chara){
    var names = this._list;
    var length = names.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var name = names[i];
        if(name[0] === chara){
            this.setTopIndex(i);
            SoundManager.playCursor();
            return;
        }
    }


    for(var i = 0; i<length; i=(i+1)|0){
        var name = names[i];
        if(name[0].toLowerCase() === chara
            || name[0].toUpperCase() === chara)
        {
            this.setTopIndex(i);
            SoundManager.playCursor();
            return;
        }
    }
};







//=============================================================================
// GuideSprite
//=============================================================================
function GuideSprite(){
    this.initialize.apply(this, arguments);
}
GuideSprite.prototype = Object.create(Sprite.prototype);
GuideSprite.prototype.constructor = GuideSprite;

GuideSprite.LINE_HEIGHT = 20;
GuideSprite.prototype.initialize = function() {
    Sprite.prototype.initialize.call(this);

    var bitmap = new Bitmap(256,512);
    this.bitmap = bitmap;
    this._texts = null;
    this._folding = false;
    this._backColor = 'rgba(0,0,0,0.5)';

    bitmap.fontSize = GuideSprite.LINE_HEIGHT - 4;
    bitmap.outlineWidth = 3;
    bitmap.outlineColor = 'black';
};

GuideSprite.prototype.setTexts = function(texts,color='rgba(0,0,0,0.5)'){
    this._backColor = color;
    if(!texts){
        this._texts = null;
        this.vibible = false;
        return;
    }else if(this._texts && texts.equals(this._texts)){
        return;
    }else{
        this._texts = texts;
        if(!this._folding){
            this.refreshTexts(texts);
        }
    }
};

GuideSprite.prototype.refreshTexts = function(texts){
    var bitmap = this.bitmap;
    bitmap.clear();

    var lineHeight = GuideSprite.LINE_HEIGHT;
    var x = 0;
    var y = 0;


    var length = texts.length;
    for(var i = 0; i<length; i=(i+1)|0){
        var text = texts[i];
        if(!text){
            y += Math.floor(lineHeight/2);
            continue;
        }

        var textWidth = bitmap.measureTextWidth(text);
        textWidth = (textWidth+4).clamp(0,bitmap.width);
        bitmap.fillRect(0,y,textWidth,lineHeight,this._backColor);

        bitmap.drawText(text,x+2,y,bitmap.width-4,lineHeight,'left');

        y += lineHeight;
    }

    this.x = 0;
    this.y = Graphics.height - y - 10;
};

GuideSprite.prototype.update = function(){
    Sprite.prototype.update.call(this);
    if(Input.isTriggered(guideKey)){
        SoundManager.playCursor();
        if(this._folding){
            this.open();
        }else{
            this.fold();
        }
    }
};

GuideSprite.prototype.open = function(){
    if(!this._folding)return;
    this._folding = false;

    if(this._texts){
        this.refreshTexts(this._texts);
    }
};

GuideSprite.FOLDING_TEXTS = ['G：ガイドを表示'];
GuideSprite.prototype.fold = function(){
    if(this._folding)return;
    this._folding = true;

    this.refreshTexts(GuideSprite.FOLDING_TEXTS);
};






//=============================================================================
// ColorPicker
//=============================================================================
function ColorPicker(){
    this.initialize.apply(this, arguments);
};
ColorPicker.colorWithHsv = function(h,s,v){
    var max = v;
    var min = max-((s/255)*max);
    var r,g,b;
    if(h<=60){
        r = max;
        g = (h/60)*(max-min)+min;
        b = min;
    }else if(h<=120){
        r = ((120-h)/60)*(max-min)+min;
        g = max;
        b = min;
    }else if(h<=180){
        r = min;
        g = max;
        b = ((h-120)/60)*(max-min)+min;
    }else if(h<=240){
        r = min;
        g = ((240-h)/60)*(max-min)+min;
        b = max;
    }else if(h<=300){
        r = ((h-240)/60)*(max-min)+min;
        g = min;
        b = max;
    }else{
        r = max;
        g = min;
        b = ((360-h)/60)*(max-min)+min;
    }
    r = Math.round(r).toString(16);
    g = Math.round(g).toString(16);
    b = Math.round(b).toString(16);
    if(r.length===1)r='0'+r;
    if(g.length===1)g='0'+g;
    if(b.length===1)b='0'+b;
    var color = '#'+r+g+b;
    return color;
};

ColorPicker.HUE_WIDTH = 20;
ColorPicker.MARGIN = 3;

ColorPicker.prototype = Object.create(PIXI.Container.prototype);
ColorPicker.prototype.constructor = ColorPicker;
ColorPicker.prototype.initialize = function(size){
    PIXI.Container.call(this);

    this._size = size;

    this._hue = -1;
    this._saturation = -1;
    this._value = -1;
    this._color = null;

    this._touchingHue = false;
    this._touchingSv = false;

    var margin = ColorPicker.MARGIN;
    var hueWidth = ColorPicker.HUE_WIDTH;
    var totalWidth = margin*3 + size + hueWidth;
    var totalHeight = margin*2 + size;

    var bitmap,sprite;

    //this > backBitmap
    bitmap = new Bitmap(16,16);
    bitmap.fillAll('rgba(0,0,0,0.5)');
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    sprite.scale.set(totalWidth/16,totalHeight/16);
    this._backSprite = sprite;


    //pickerSprite
    bitmap = new Bitmap(size,size);
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    sprite.x = margin;
    sprite.y = margin;
    this._pickerSprite = sprite;
    this.bitmap = bitmap;

    //huePicker
    bitmap = new Bitmap(hueWidth,size);
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    sprite.x = margin*2 + size;
    sprite.y = margin;
    this._huePicker = sprite;

    //pointer
    bitmap = new Bitmap(16,16);
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    sprite.anchor.set(0.5,0.5);
    this._pointer = sprite;
    var ctx = bitmap._context;
    ctx.beginPath();
    ctx.arc(8,8,6,0,360*Math.PI/180,false);
    ctx.fillStyle = 'rgb(255,255,255)';
    ctx.fill();
    ctx.beginPath();
    ctx.arc(8,8,3,0,360*Math.PI/180,false);
    ctx.globalCompositeOperation = "destination-out";
    ctx.fill();

    //huePointer
    var lineWidth = 2;
    var spaceHeight = 2;
    bitmap = new Bitmap(hueWidth+lineWidth*2,spaceHeight+lineWidth*2);
    sprite = new Sprite(bitmap);
    this.addChild(sprite);
    sprite.anchor.set(0.5,0.5);
    this._huePointer = sprite;
    bitmap.fillAll('black');
    bitmap.clearRect(lineWidth,lineWidth,bitmap.width-lineWidth*2,bitmap.height-lineWidth*2);


    this.setupHuePicker();
    this.setColor('rgb(255,255,255)');
};
ColorPicker.prototype.setupHuePicker = function(){
    var bitmap = this._huePicker.bitmap;
    var width = bitmap.width;
    var height = bitmap.height;

    var s = 255;
    var v = 255;
    for(var y=0; y<height; y=(y+1)|0){
        var h = 360*(y/height);
        var color = ColorPicker.colorWithHsv(h,s,v);
        bitmap.fillRect(0,y,width,1,color);
    }
};
ColorPicker.prototype.setupPallete = function(h){
    var bitmap = this._pickerSprite.bitmap;
    bitmap.clear();

    var width = this.width;
    var height = this.height;

    var r,g,b;
    for(var x=0; x<width; x=(x+1)|0){
        var s = 255*x/width;
        for(var y=0; y<height; y=(y+1)|0){
            var v = 255*y/height;
            var color = ColorPicker.colorWithHsv(h,s,v);
            bitmap.fillRect(x,height-y-1,1,1,color);
        }
    }
};
ColorPicker.prototype.setColor = function(color){
    var r,g,b;
    if(Array.isArray(color)){  
        r = color[0];
        g = color[1];
        b = color[2];
    }else if(color.indexOf('rgb')!==0){
        if(color[0] == "#"){
            color = color.substr(1);
        }else if(color.indexOf("0x")===0){
            color = color.substr(2);
        }
        if(color.length == 8){
            color = color.substr(2);
        }
        r = parseInt(color.substr(0, 2), 16);
        g = parseInt(color.substr(2, 2), 16);
        b = parseInt(color.substr(4, 2), 16);
    }else{
        var args = color.match(/\((.+)\)/)[1].split(',');
        r = Number(args[0]);
        g = Number(args[1]);
        b = Number(args[2]);
    }

    var h,s,v;
    var max = Math.max(r,g,b);
    var min = Math.min(r,g,b);
    if(r===g && g===b){
        h = Math.max(0,this._hue);
    }else if(r>=g && r>=b){
        h = 60*(g-b)/(max-min);     
    }else if(g>=r && g>=b){
        h = 60*(b-r)/(max-min)+120;
    }else{
        h = 60*(r-g)/(max-min)+240;
    }

    s = (max-min)/max*255;
    v = max;

    this.setHue(h);
    this.setSV(s,v);
};
ColorPicker.prototype.updateResultColor = function(){
    this._color = ColorPicker.colorWithHsv(this._hue,this._saturation,this._value);

    this.applyData();
};
ColorPicker.prototype.color = function(){
    return this._color;
};
ColorPicker.prototype.setHue = function(h){
    h = h.clamp(0,360);
    if(this._hue === h)return;

    var dh = h-this._hue;
    this._hue = h;
    this.setupPallete(this._hue);

    var sprite = this._huePicker;
    var pointer = this._huePointer;
    pointer.x = sprite.x+sprite.width/2;
    pointer.y = sprite.y+sprite.height*h/360;

    this.updateResultColor();
};
ColorPicker.prototype.setSV = function(s,v){
    if(this._saturation===s && this._value===v)return;

    this._saturation = s;
    this._value = v;

    var margin = ColorPicker.MARGIN
    var size = this._size;

    var pointer = this._pointer;
    pointer.x = margin+Math.round((s/255)*size);
    pointer.y = margin+Math.round(size-(v/255)*size-1);

    this.updateResultColor();
};
ColorPicker.prototype.update = function(){
    if(!this.visible){
        this._touchingHue = false;
        this._touchingSv = false;
        return;
    }

    if(Input._latestButton){
        this.processInput();
    }

    if(!TouchInput.isTriggered() && !TouchInput.isPressed()){
        this._touchingHue = false;
        this._touchingSv = false;
        return;
    }

    var x = TouchInput.x-this.x;
    var y = TouchInput.y-this.y;
    var dx,dy,touchInside;

    var hPicker = this._huePicker;
    dx = x-hPicker.x;
    dy = y-hPicker.y;

    touchInside = (dx>=0 && dx<=hPicker.width && dy>=0 && dy<=hPicker.height);
    if(this._touchingHue || (!this._touchingSv&&touchInside)){
        dy = dy.clamp(0,hPicker.height-1);
        var hue = Math.round(dy/(hPicker.height-1)*360);
        this.setHue(hue);
        this._touchingHue = true;
        return;
    }

    var svPicker = this._pickerSprite;
    dx = x-svPicker.x;
    dy = y-svPicker.y;
    touchInside = (dx>=0 && dx<=svPicker.width && dy>=0 && dy<=svPicker.height);
    if(this._touchingSv || (!this._touchingHue&&touchInside)){
        dx = dx.clamp(0,svPicker.width-1);
        dy = dy.clamp(0,svPicker.height-1);
        var s = Math.round(dx/(svPicker.width-1)*255);
        var v = Math.round((svPicker.height-1-dy)/(svPicker.height-1)*255);
        this.setSV(s,v);
        this._touchingSv = true;
        return;
    }

    this.end();
};


ColorPicker.prototype.processInput = function(){
    if(Input.isTriggered('ok')||Input.isTriggered('cancel')){
        this.end();
    };
};

ColorPicker.prototype.startPicking = function(owner,color){
    this.visible = true;
    this._owner = owner;

    Input.clear();
    TouchInput.clear();

    this.setColor(color);
};
ColorPicker.prototype.end = function(){
    if(this._owner){
        this._owner.didEndPicking();
    }
    this._owner = null;
    this.visible = false;

    SoundManager.playCancel();
    Input.clear();
    TouchInput.clear();
};

ColorPicker.prototype.applyData = function(){
    if(!this._owner)return;

    var color = this._color;
    if(color[0] == "#"){
        color = color.substr(1);
    }else if(color.indexOf("0x")===0){
        color = color.substr(2);
    }
    if(color.length == 8){
        color = color.substr(2);
    }
    var r = parseInt(color.substr(0, 2), 16);
    var g = parseInt(color.substr(2, 2), 16);
    var b = parseInt(color.substr(4, 2), 16);

    this._owner.setColor(r,g,b);
};



})();










// --------------------------------------------------
// easing.js v0.5.4
// Generic set of easing functions with AMD support
// https://github.com/danro/easing-js
// This code may be freely distributed under the MIT license
// http://danro.mit-license.org/
// --------------------------------------------------
// All functions adapted from Thomas Fuchs & Jeremy Kahn
// Easing Equations (c) 2003 Robert Penner, BSD license
// https://raw.github.com/danro/easing-js/master/LICENSE
// --------------------------------------------------

TRP_CORE.easing = {
  linear: function(pos) {return pos},

  easeIn: function(pos) {return Math.pow(pos, 2);},
  easeOut: function(pos) {return -(Math.pow((pos-1), 2) -1);},
  easeInOut: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,2);
    return -0.5 * ((pos-=2)*pos - 2);
  },

  quadIn: function(pos) {return Math.pow(pos, 2);},
  quadOut: function(pos) {return -(Math.pow((pos-1), 2) -1);},
  quadInOut: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,2);
    return -0.5 * ((pos-=2)*pos - 2);
  },

  cubicIn: function(pos) {return Math.pow(pos, 3);},
  cubicOut: function(pos) {return (Math.pow((pos-1), 3) +1);},
  cubicInOut: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,3);
    return 0.5 * (Math.pow((pos-2),3) + 2);
  },

  quartIn: function(pos) {return Math.pow(pos, 4);},
  quartOut: function(pos) {return -(Math.pow((pos-1), 4) -1);},
  quartInOut: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,4);
    return -0.5 * ((pos-=2)*Math.pow(pos,3) - 2);
  },

  quintIn: function(pos) {return Math.pow(pos, 5);},
  quintOut: function(pos) {return (Math.pow((pos-1), 5) +1);},
  quintInOut: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,5);
    return 0.5 * (Math.pow((pos-2),5) + 2);
  },

  sineIn: function(pos) {return -Math.cos(pos * (Math.PI/2)) + 1;},
  sineOut: function(pos) {return Math.sin(pos * (Math.PI/2));},
  sineInOut: function(pos) {
    return (-0.5 * (Math.cos(Math.PI*pos) -1));
  },

  expoIn: function(pos) {return (pos===0) ? 0 : Math.pow(2, 10 * (pos - 1));},
  expoOut: function(pos) {return (pos===1) ? 1 : -Math.pow(2, -10 * pos) + 1;},
  expoInOut: function(pos) {
    if(pos===0) return 0;
    if(pos===1) return 1;
    if((pos/=0.5) < 1) return 0.5 * Math.pow(2,10 * (pos-1));
    return 0.5 * (-Math.pow(2, -10 * --pos) + 2);
  },

  circIn: function(pos) {return -(Math.sqrt(1 - (pos*pos)) - 1);},
  circOut: function(pos) {return Math.sqrt(1 - Math.pow((pos-1), 2));},
  circInOut: function(pos) {
    if((pos/=0.5) < 1) return -0.5 * (Math.sqrt(1 - pos*pos) - 1);
    return 0.5 * (Math.sqrt(1 - (pos-=2)*pos) + 1);
  },

  bounceOut: function(pos) {
    if ((pos) < (1/2.75)) {
      return (7.5625*pos*pos);
    } else if (pos < (2/2.75)) {
      return (7.5625*(pos-=(1.5/2.75))*pos + 0.75);
    } else if (pos < (2.5/2.75)) {
      return (7.5625*(pos-=(2.25/2.75))*pos + 0.9375);
    } else {
      return (7.5625*(pos-=(2.625/2.75))*pos + 0.984375);
    }
  },

  backIn: function(pos) {
    return (pos)*pos*((1.70158+1)*pos - 1.70158);
  },
  backOut: function(pos) {
    return (pos=pos-1)*pos*((2.70158)*pos + 1.70158) + 1;
  },
  backInOut: function(pos) {
    var s = 1.70158;
    if((pos/=0.5) < 1) return 0.5*(pos*pos*(((s*=(1.525))+1)*pos -s));
    return 0.5*((pos-=2)*pos*(((s*=(1.525))+1)*pos +s) +2);
  },

  elastic: function(pos) {return -1 * Math.pow(4,-8*pos) * Math.sin((pos*6-1)*(2*Math.PI)/2) + 1;},

  swingFromTo: function(pos) {
    var s = 1.70158;
    return ((pos/=0.5) < 1) ? 0.5*(pos*pos*(((s*=(1.525))+1)*pos - s)) :
    0.5*((pos-=2)*pos*(((s*=(1.525))+1)*pos + s) + 2);
  },

  swingFrom: function(pos) {
    var s = 1.70158;
    return pos*pos*((s+1)*pos - s);
  },

  swingTo: function(pos) {
    var s = 1.70158;
    return (pos-=1)*pos*((s+1)*pos + s) + 1;
  },

  bounce: function(pos) {
    if (pos < (1/2.75)) {
      return (7.5625*pos*pos);
    } else if (pos < (2/2.75)) {
      return (7.5625*(pos-=(1.5/2.75))*pos + 0.75);
    } else if (pos < (2.5/2.75)) {
      return (7.5625*(pos-=(2.25/2.75))*pos + 0.9375);
    } else {
      return (7.5625*(pos-=(2.625/2.75))*pos + 0.984375);
    }
  },

  bouncePast: function(pos) {
    if (pos < (1/2.75)) {
      return (7.5625*pos*pos);
    } else if (pos < (2/2.75)) {
      return 2 - (7.5625*(pos-=(1.5/2.75))*pos + 0.75);
    } else if (pos < (2.5/2.75)) {
      return 2 - (7.5625*(pos-=(2.25/2.75))*pos + 0.9375);
    } else {
      return 2 - (7.5625*(pos-=(2.625/2.75))*pos + 0.984375);
    }
  },

  easeFromTo: function(pos) {
    if ((pos/=0.5) < 1) return 0.5*Math.pow(pos,4);
    return -0.5 * ((pos-=2)*Math.pow(pos,3) - 2);
  },

  easeFrom: function(pos) {
    return Math.pow(pos,4);
  },

  easeTo: function(pos) {
    return Math.pow(pos,0.25);
  }
};









