//=============================================================================
// TRP_ProgressMovieTemplate.js
//=============================================================================
// PluginCommand
// ProgressMovie
// Play Command


/*:
 * @target MZ
 * @plugindesc 進捗動画テンプレート<開発用>
 * @author Thirop
 * @base TRP_Movies
 * @base TRP_CORE
 * @orderAfter TRP_CORE
 * @orderAfter TRP_Movies
 * @help
 * 以下のプラグインより下に配置
 * ・TRP_CORE.js
 * ・TRP_Movies.js
 *
 *
 *
 * @command prepare
 * @text [設定]進捗動画の開始準備
 * @desc 進捗動画の開始準備
 * 
 * @arg width
 * @text ウィンドウリサイズ幅
 * @desc ウィンドウリサイズ幅。0でリサイズなし
 * @type number
 * @default 0
 * @min 0
 *
 * @arg height
 * @text ウィンドウリサイズ高さ
 * @desc ウィンドウリサイズ高さ。0でリサイズなし
 * @type number
 * @default 0
 * @min 0
 *
 *
 * @command messageWindow
 * @text [設定]メッセージウィンドウ設定
 * @desc メッセージウィンドウの設定
 * 
 * @arg rows
 * @text 行数
 * @desc 行数
 * @type number
 * @default 2
 *
 * @arg paddingTop
 * @text 上部の余白
 * @desc 上部の余白ピクセル。-1で変更なし
 * @type number
 * @default -1
 * @min -1
 *
 * @arg paddingSide
 * @text 左右の余白
 * @desc 左右の余白ピクセル。-1で変更なし
 * @type number
 * @default -1
 * @min -1
 *
 * @arg paddingBottom
 * @text 下部の余白
 * @desc 下部の余白ピクセル。-1で変更なし
 * @type number
 * @default -1
 * @min -1
 *
 * @arg hideNameBox
 * @text ネームボックス非表示
 * @desc MZ用設定。ONにするとネームボックス非表示
 * @default false
 * @type boolean
 *
 * @arg image
 * @text ウィンドウ画像名
 * @desc 1枚絵形式のウィンドウ画像ファイル名。画像は進捗動画フォルダに配置。省略時は使用なし。
 * @type string
 *
 *
 *
 *
 * @command messageText
 * @text [設定]メッセージテキスト設定
 * @desc メッセージウィンドウのテキスト表示設定。
 * 
 * @arg fontSize
 * @text フォントサイズ
 * @desc フォントサイズ。0で変更なし
 * @type number
 * @default 36
 * @min 0
 *
 * @arg textColor
 * @text 文字色
 * @desc 文字色。rgb(赤,緑,青)でそれぞれ0~255の値で設定。未設定で変更なし。
 * @type string
 * @default rgb(255,255,255)
 *
 * @arg outlineWidth
 * @text 枠線の幅
 * @desc 枠線の幅
 * @type outlineWidth
 * @default 6
 *
 * @arg outlineColor
 * @text 枠線の色
 * @desc 枠線の色。rgb(赤,緑,青)でそれぞれ0~255の値で設定。未設定で変更なし。
 * @default rgb(0,100,100)
 *
 * @arg lineMargin
 * @text 行間の幅
 * @desc 行間の幅ピクセル数
 * @default 10
 * @type number
 *
 *
 * @command mainMovie
 * @text [設定]メインムービー
 * @desc メインムービーの表示設定
 * 
 * @arg lrRatio
 * @text 左右余白の割合
 * @desc 横幅に余白がある場合の左右の余白幅の割合。(右に立ち絵表示するなら左に寄せるなど)。デフォルトは「1:1」で中央
 * @type string
 * @default 1:1
 *
 * @arg tbRatio
 * @text 上下余白の割合
 * @desc 縦幅に余白がある場合の上下の余白幅の割合。(メッセージに被せないよう上に寄せるなど)。デフォルトは「0:1」で上詰め
 * @type string
 * @default 0:1
 *
 * @arg resizeMode
 * @text リサイズモード
 * @desc 動画を自動でリサイズ
 * @default fit
 * @type select
 * @option none/リサイズなし
 * @value none
 * @option fit/画面にフィット(余白あり)
 * @value fit
 * @option fitWidth/横幅をフィット
 * @value fitWidth
 * @option fitHeight/高さをフィット
 * @value fitHeight
 * @option fill/画面を埋める(はみ出しあり)
 * @value fill
 *
 * @arg fitRate
 * @text フィット率
 * @desc リサイズモードでfit,fitWidth,fitHeight指定時のフィット率(%)
 * @default 100
 * @type number
 * @min 1
 * @decimals 1
 *
 * @arg loop
 * @text 動画をループ
 * @desc 動画をループ。OFFにすると最終フレームのまま停止
 * @type boolean
 * @default true
 *
 *
 * @command backMovie
 * @text [設定]背景ムービー
 * @desc 背景動画の設定。表示する必要がない場合は呼び出し不要
 * 
 * @arg blur
 * @text ブラーの大きさ
 * @desc ブラー（にじみ）の大きさ
 * @type number
 * @default 10
 *
 * @arg opacity
 * @text 不透明度/オパシティ
 * @desc 不透明度/オパシティの値(0~255)
 * @type number
 * @default 100
 *
 *
 * @command skit
 * @text [設定]TRP_Skit/立ち絵
 * @desc TRP_Skit(MZ)用の立ち絵設定
 * 
 * @arg scale
 * @text 拡大率
 * @desc 立ち絵の拡大率(%)
 * @type number
 * @default 100
 * @min 1
 * @decimals 1
 *
 * @arg deltaBaseOffsetY
 * @text Y座標調整
 * @desc 立ち絵Y座標の調整(ピクセル)
 * @default 0
 * @min -9999
 * @type number
 *
 * @arg positions
 * @text デフォ表示位置
 * @desc デフォルトの立ち絵表示位置(0~10)。自動表示有効時はカンマ「,」区切りで指定「8.5,1.5」など
 * @default 8.5
 * @type number
 * @min -10
 * @decimals 2
 *
 * @arg zOrderFront
 * @text ピクチャの前に変更
 * @desc 重なり順をピクチャの前に変更
 * @type boolean
 * @default true
 *
 *
 *
 *
 *
 * @command play
 * @text [制御]ムービー再生
 * @desc (次の)ムービーの再生開始
 * 
 * @arg name
 * @text 動画ファイル名
 * @desc 動画ファイル名(兼ムービー管理ID)。動画はドラッグ＆ドロップで取り込むか、設定した進捗動画フォルダに配置
 * @type string
 * @default 
 *
 * @arg duration
 * @text フェードフレーム数
 * @desc クロスフェードするフレーム数
 * @default 10
 * @type number
 * @min 0
 *
 * @arg delay
 * @text ディレイフラグ
 * @desc ONにすると、次の動画が完全に表示されてからフェードアウト
 * @type boolean
 * @default false
 *
 *
 * @command waitInput
 * @text [制御]インプット待ち
 * @desc インプット(OK/タッチ)までウェイト。メッセージをいったん閉じて動画を見せたいときなど
 *
 *
 *
 * =====================================================
 *
 *
 *
 * @param command
 * @text コマンド名(MV)
 * @desc MV形式プラグインコマンドのコマンド名
 * @default pv
 * @type string
 *
 * @param enableDragAndDrop
 * @text ドラッグ&ドロップ有効
 * @desc ONにすると「準備コマンド」後にドラッグ&ドロップでの動画の取り込みを有効
 * @type boolean
 * @default true
 *
 * @param movieDir
 * @text 進捗動画フォルダ
 * @desc 読み込む進捗動画のフォルダ。ドラッグ＆ドロップ有効時にはこのフォルダへコピペ。省略時は「movies」
 * @default _dev/progressMovies
 * @type string
 *
 * @param pictureId
 * @text 専有ピクチャ開始番号
 * @desc プラグイン内で使用するピクチャID。連続して10個使用。(メイン動画は4番目)
 * @default 1
 * @type number
 *
 *
 *
 *
 *
 * 
 */
//============================================================================= 


(function(){
'use strict';

function evalBoolean(value){
	return value==='true'||value===true;
};


var isMZ = Utils.RPGMAKER_NAME==="MZ";
var pluginName = 'TRP_ProgressMovieTemplate';
var parameters = PluginManager.parameters(pluginName);
var enableDragAndDrop = evalBoolean(parameters.enableDragAndDrop);
var movieDir = parameters.movieDir||'movies/';
if(movieDir[movieDir.length-1]!=='/'){
	movieDir = movieDir+'/';
}
var pictureId0 = Number(parameters.pictureId)||0;


var MovieManager = TRP_MovieManager;
var Movie = MovieManager.Movie;
var _Dev = TRP_CORE.DevFuncs;

var ProgressMovie = MovieManager.ProgressMovie = function(){};






//=============================================================================
// PluginCommand
//=============================================================================
var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command,args){
	if(command === parameters.command){
		ProgressMovie.pluginCommand(args,this);
	}else{
		_Game_Interpreter_pluginCommand.call(this,...arguments);
	}
};

if(isMZ){
	['prepare','messageText','messageWindow','mainMovie','backMovie','skit','play','waitInput'].forEach(command=>{
		PluginManager.registerCommand(pluginName,command,function(args){
			var argsArr = Object.values(args)
			argsArr.unshift(command);
			ProgressMovie.pluginCommand(argsArr,this);
		});
	});
};

ProgressMovie.pluginCommand = function(args,interpreter){
	var name = args[0];
	var command = "processCommand"+name[0].toUpperCase()+name.substring(1)
	this[command](args,interpreter);
};





//=============================================================================
// ProgressMovie
//=============================================================================
ProgressMovie._setup = false;
ProgressMovie.processCommandPrepare = function(args,interpreter){
	var width = Number(args[1])||0;
	var height = Number(args[2])||0;
	if(width || height){
		_Dev.resizeWindow(width,height);
	}
	if(!this._setup){
		this.setup();
	}

	SoundManager.playOk();
};
ProgressMovie.setup = function(){
	this._setup = true;

	/* enableDragAndDrop
	===================================*/
	if(enableDragAndDrop){
		this.enableDragAndDrop();
	}

	/* register update
	===================================*/
	var scene = SceneManager._scene;
	var update = scene.update;
	scene.update = function(){
		update.call(scene);
		ProgressMovie.update();
	}


	/* register waitTouch mode
	===================================*/
	var _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
	Game_Interpreter.prototype.updateWaitMode = function(){
		if(this._waitMode==='trpWaitInput'){
			if(Input.isTriggered('ok') || TouchInput.isTriggered()){
				this._waitMode = '';
				Input.clear();
				TouchInput.clear();
			}
			return !!this._waitMode;
		}else{
			return _Game_Interpreter_updateWaitMode.call(this,...arguments);
		}
	}
}


ProgressMovie.enableDragAndDrop = function(){
	_Dev.showTempAlert('《ドラッグ＆ドロップ有効化！》');

	var div = document.createElement('div');
	document.body.appendChild(div);
	div.style.width = "100%";
    div.style.height = "100%";
    div.style.margin = "0px";
    div.style.padding = "0px";
    div.style.left = "0";
    div.style.top = "0";
    div.style.zIndex = "999";
    div.style.position = "absolute";

    div.addEventListener('dragover',event=>{
    	event.preventDefault();
    });
	div.addEventListener('drop',event=>{
		event.preventDefault();

		var files = event.dataTransfer.files;
		if(!files || !files.length)return;

		for(const file of files){
			if(file.type.indexOf('video/')<0)return;

			var reader = new FileReader();
			reader.onloadend = function(file,reader){
				var filePath = movieDir+file.name;
				_Dev.ensureDirectoriesWithFilePath(filePath);

				var buffer = new Uint8Array(reader.result);
				_Dev._saveFile(buffer,filePath);
				SoundManager.playSave();
				_Dev.showTempText('%1の取り込み完了!'.format(file.name));
			}.bind(this,file,reader);
			reader.readAsArrayBuffer(file);
		}
	});

};



/* update
===================================*/
ProgressMovie.update = function(){
	if(this._fadeOutCount>0){
		this._fadeOutCount -= 1;
		if(this._fadeOutCount<=0){
			this.processFadeOutLastMoviePictures();
		}
	}
	if(this._clearLastCount>0){
		this._clearLastCount -= 1;
		if(this._clearLastCount<=0){
			this.processClearLastMovie();
		}
	}
}



/* messageWindow
===================================*/
ProgressMovie.processCommandMessageWindow = function(args,interpreter){
	var i = 1;
	var rows = Number(args[i++]);
	var padTop = Number(args[i++]);
	var padSide = Number(args[i++]);
	var padBottom = Number(args[i++]);
	var hideNameBox = evalBoolean(args[i++]);
	var image = args[i++];

	var padding = SceneManager._scene._messageWindow._padding;
	if(padTop<0)padTop = padding;
	if(padSide<0)padSide = padding;
	if(padBottom<0)padBottom = padding;

	if(image){
		Window_Message.prototype._createAllParts = function() {
			Window.prototype._createAllParts.call(this);
			this.createTrpBackSprite();
		};
		Window_Message.prototype.createTrpBackSprite = function(){
			var sprite = new Sprite();
			this._trpBackSprite = sprite;
			sprite.bitmap = ImageManager.loadBitmap(movieDir, image.replace('.png',''));

			if(isMZ){
				this._container.addChild(this._trpBackSprite);
			}else{
				this._windowSpriteContainer.addChild(sprite);
			}
		}

		if(isMZ){
			Window_Message.prototype._updateFrame = function() {
				if(this._trpBackSprite){
					this._frameSprite.visible = false;
					this._contentsBackSprite.visible = false;
					this._backSprite.visible = false;
					this._trpBackSprite.visible = this.frameVisible;
				}else{
					Window.prototype._updateFrame.call(this);
				}
			};

		}else{
			Object.defineProperty(Window.prototype, 'opacity', {
			    get: function() {
			        return this._windowSpriteContainer.alpha * 255;
			    },
			    set: function(value) {
			    	this._windowFrameSprite.alpha = 0;
			    	this._windowBackSprite.alpha = 0;
			        this._windowSpriteContainer.alpha = value.clamp(0, 255) / 255;
			    },
			    configurable: true_windowFrameSprite
			});
		}
	}

	

	if(isMZ){
		/* MZ
		===================================*/
		Scene_Message.prototype.messageWindowRect = function(){
			const ww = Graphics.boxWidth;
		    const wh = Window_Message.prototype.fittingHeight(2)+8;
		    const wx = (Graphics.boxWidth - ww) / 2;
		    const wy = 0;
		    return new Rectangle(wx, wy, ww, wh);
		};

		Object.defineProperty(Window_Message.prototype, "innerWidth", {
		    get: function() {
		        return Math.max(0, this._width - padSide * 2);
		    },
		    configurable: true
		});
		Object.defineProperty(Window.prototype, "innerHeight", {
		    get: function() {
		        return Math.max(0, this._height - padTop - padBottom);
		    },
		    configurable: true
		});

		Window_Message.prototype._updateClientArea = function() {
		    this._clientArea.move(padTop, padSide);
		    this._clientArea.x = padSide - this.origin.x;
		    this._clientArea.y = padTop - this.origin.y;
		    if (this.innerWidth > 0 && this.innerHeight > 0) {
		        this._clientArea.visible = this.isOpen();
		    } else {
		        this._clientArea.visible = false;
		    }
		};

		Window_Message.prototype.fittingHeight = function(numLines) {
		    return numLines * this.itemHeight() + padTop + padBottom;
		};

		if(hideNameBox){
			Window_NameBox.prototype.setName = function(){};
		}
	}else{
		/* MV
		===================================*/
		Window_Message.prototype.createBackImage = function(){
			var src = ImageManager.loadBitmap('img/_dev/devMovie/', 'message_window')
			var sprite = SpriteManager.simpleSprite(src);
			this._backSprite = sprite;
			this._windowSpriteContainer.addChildAt(sprite,0);
		};

		Window_Message.prototype.refreshContentsSpritePosition = function(){
			this._windowContentsSprite.x = padSide;
			this._windowContentsSprite.y = padTop;
		};

		Window_Message.prototype._refreshContents = function() {
		    this._windowContentsSprite.move(padSide, padTop);
		};

		Window_Message.prototype.windowHeight = function() {
			// return 147
			return 134;
		};
		Window_Message.prototype.windowWidth = function() {
			return Graphics.width;
		};
		Window_Message.prototype.contentsWidth = function() {
		    return this.width - padSide * 2;
		};
		Window_Message.prototype.numVisibleRows = function(){
			return rows;
		};
		Window_Message.prototype.fittingHeight = function(numLines) {
		    return numLines * this.lineHeight() + padTop + padBottom;
		};
		Window_Message.prototype.updatePlacement = function() {
		    this.y = (Graphics.height - this.height);
		};
	}
	this.recreateMessageWindow();
};

ProgressMovie.recreateMessageWindow = function(){
	var scene = SceneManager._scene;
	if(scene._messageWindow){
		scene._messageWindow.parent.removeChild(scene._messageWindow);

		if(!isMZ){
			scene._messageWindow.subWindows().forEach(function(window){
		        if(window.parent)window.parent.removeChild(window);
		    }, this);
		}

	    scene._messageWindow = null;
	}

	scene.createMessageWindow();

	if(isMZ){
		scene.associateWindows();
	}
};


ProgressMovie.processCommandMessageText = function(args,interpreter){
	var i = 1;
	var fontSize = Number(args[i++])||0;
	var textColor = args[i++];
	var outlineWidth = Number(args[i++])||0;
	var outlineColor = args[i++];
	var lineMargin = Number(args[i++])||0;
	// var font = args[i++];


	var _Window_Message_resetFontSettings = Window_Message.prototype.resetFontSettings;
	Window_Message.prototype.resetFontSettings = function(){
		_Window_Message_resetFontSettings.call(this,...arguments);	
		// if(font){
		// 	this.contents.fontFace = font;
		// }
		if(fontSize){
			this.contents.fontSize = fontSize;
		}
		if(textColor){
			this.contents.textColor = textColor;
		}
		this.contents.outlineWidth = outlineWidth;
		if(outlineColor){
			this.contents.outlineColor = outlineColor;
		}

		Window_Base.prototype.lineHeight = function() {
		    return fontSize+Math.ceil(outlineWidth/2)+lineMargin;
		};
	};

	if(isMZ){
		Game_System.prototype.mainFontSize = function() {
		    return fontSize;
		};
	}else{
		Window_Message.prototype.standardFontSize = function(){
			return fontSize;
		}
	}

	this.recreateMessageWindow();
};


/* mainMovieSetting
===================================*/
ProgressMovie.lrRatio = 0.5;
ProgressMovie.tbRatio = 0.5;
ProgressMovie.resizeMode = 'fit';
ProgressMovie.fitRate = 1;
ProgressMovie.loop = true;
ProgressMovie.processCommandMainMovie = function(args,interpreter){
	var i = 1;
	var lrRatio = args[i++];
	var tbRatio = args[i++];
	this.resizeMode = args[i++]||'fit';
	this.fitRate = (Number(args[i++])||100)/100;
	this.loop = evalBoolean(args[i++]);

	var rates = lrRatio.split(':');
	this.lrRatio = Number(rates[0])/(Number(rates[0])+Number(rates[1]));
	if(isNaN(this.lrRatio))this.lrRatio = 0.5;

	var rates = tbRatio.split(':');
	this.tbRatio = Number(rates[0])/(Number(rates[0])+Number(rates[1]));
	if(isNaN(this.tbRatio))this.tbRatio = 0.5;

};

/* backMovieSetting
===================================*/
ProgressMovie.useBackMovie = false;
ProgressMovie.backBlur = 10;
ProgressMovie.backOpacity = 150;
ProgressMovie.processCommandBackMovie = function(args,interpreter){
	this.useBackMovie = true;

	var i = 1;
	this.backBlur = Number(args[i++])||0;
	this.backOpacity = Number(args[i++])||0;
};




/* Skit Setting
===================================*/
ProgressMovie._skitStartAnimation = null;
ProgressMovie.processCommandSkit = function(args,interpreter){
	var i = 1;
	var scale = (Number(args[i++])||100)/100;
	var deltaBaseOffsetY = Number(args[i++])||0;
	var positions = JSON.parse('['+args[i++]+']');
	var zOrderFront = evalBoolean(args[i]);

	var skitParams = TRP_CORE.skitParameters;
	skitParams.defaultPositionX = isNaN(positions[0])?8:positions[0];
	skitParams.bustsScale = scale*100;
	skitParams.autoModePositions = positions;

	var autoShowParams = PluginManager.parameters('TRP_SkitExAutoShow')
	autoShowParams.positions = positions;


	var sprset = SceneManager._scene._spriteset;
	if(zOrderFront){
		var zOrder = 1;
		skitParams.zOrder = zOrder;
		sprset._bustPictureZOrder = zOrder;
		var bustsContainer = sprset._bustPictureEffectsContainer || sprset._bustPictureContainer;
		if(bustsContainer.parent){
			bustsContainer.parent.removeChild(bustsContainer);
		}
		var index = (sprset.children.indexOf(sprset._pictureContainer)+1)||0;
		sprset.addChildAt(bustsContainer,index);
	}


	ProgressMovie._skitStartAnimation = ProgressMovie._skitStartAnimation||SkitActor.prototype.startAnimation;
	SkitActor.prototype.startAnimation = function(animationId,mirror,dx,dy,scale,rotation){
		dx *= scale;
		dy *= scale;
		scale *= scale;
		ProgressMovie._skitStartAnimation.call(this,animationId,mirror,dx,dy,scale,rotation);
	};


	Skit.prototype.defaultPositionY = function(){
		var y = skitParams.baseOffsetY*scale + deltaBaseOffsetY;
		if(parameters.verticalPositionType===0){
			var messageWindow = SceneManager._scene._messageWindow;
			y += messageWindow ? messageWindow.height : 0;
		}
		return y;
	};

}









//=============================================================================
// Play Command
//=============================================================================
ProgressMovie._fadeOutCount = 0;
ProgressMovie._clearLastCount = 0;
ProgressMovie._movieId = null;
ProgressMovie._lastMovieId = null;
ProgressMovie._fadeOutDuration = 0;
ProgressMovie.processCommandPlay = function(args,interpreter){
	var i = 1;
	var name = args[i++];
	var duration = Number(args[i++])||0;
	var delay = evalBoolean(args[i]);


	/* prepare movie pictures
	===================================*/
	if(!this._moviePicturesPrepared){
		this._moviePicturesPrepared = true;
		this.prepareMoviePictures();
	}else{
		this.swapMoviePictures();
	}

	var id = name;
	if(this._lastMovieId){
		this.processClearLastMovie();
	}
	this._lastMovieId = this._movieId;
	this._movieId = id;



	/* setup movie & pictures
	===================================*/
	MovieManager.setup(id,name,movieDir,this.resizeMode,this.fitRate,this.loop,false,'map');
	var main = $gameScreen.picture(PICTURE_IDS.main);
	main._trpMovieId = id;

	var back = this.useBackMovie ? $gameScreen.picture(PICTURE_IDS.back) : null;
	if(back){
		back._trpMovieId = id;

		if(this.backBlur){
			this.applyFilterBlurToPicture(PICTURE_IDS.back,this.backBlur);
		}
	}


	/* mainPicture pos
	===================================*/
	var movie = TRP_MovieManager.movies[id];
	movie.addLoadListener(()=>{
		var w = movie.width;
		var h = movie.height;
		var mw = Graphics.width - w;
		var mh = Graphics.height - h;

		var x = 0;
		var y = 0;
		if(mw>0){
			x = mw*this.lrRatio;
		}
		if(mh>0){
			y = mh*this.tbRatio;
		}
		main._x = main._targetX = Math.floor(x+w/2);
		main._y = main._targetY = Math.floor(y+h/2);

		if(back){
			/* backPicture scale & pos
			===================================*/
			var wRate = Graphics.width/w;
			var hRate = Graphics.height/h;
			var rate = Math.max(wRate,hRate)||1;
			back._scaleX = back._targetScaleX = rate*100;
			back._scaleY = back._targetScaleY = rate*100;
		}
	});


	/* fade animation
	===================================*/
	this.fadePicture(PICTURE_IDS.main,duration,255);
	if(this.useBackMovie){
		this.fadePicture(PICTURE_IDS.back,duration,this.backOpacity);
	}

	this._fadeOutCount = delay ? duration : 0;
	this._fadeOutDuration = duration;
	if(!delay){
		this.processFadeOutLastMoviePictures();
	}
};

ProgressMovie.processFadeOutLastMoviePictures = function(){
	var duration = this._fadeOutDuration;
	this.fadePicture(PICTURE_IDS.lastMain,duration,0);
	if(this.useBackMovie){
		this.fadePicture(PICTURE_IDS.lastBack,duration,0);
	}

	this._clearLastCount = duration+1;
};
ProgressMovie.processClearLastMovie = function(){
	if(this._lastMovieId){
		TRP_MovieManager.clear(this._lastMovieId);
	}
	this._lastMovieId = null;
};


/* waitInput
===================================*/
ProgressMovie.processCommandWaitInput = function(args,interpreter){
	interpreter._waitMode = 'trpWaitInput';
};




/* Pictures
===================================*/
var PICTURE_IDS = {
	lastBack:pictureId0+0,
	back:pictureId0+1,
	lastMain:pictureId0+2,
	main:pictureId0+3,
};
ProgressMovie._moviePicturesPrepared = false;
ProgressMovie.prepareMoviePictures = function(){
	var pictureIds = [];
	var origin;
	if(this.useBackMovie){
		origin = 1;
		pictureIds.push([PICTURE_IDS.lastBack,origin]);
		pictureIds.push([PICTURE_IDS.back,origin]);
	}
	origin = 1;
	pictureIds.push([PICTURE_IDS.lastMain,origin]);
	pictureIds.push([PICTURE_IDS.main,origin])


	pictureIds.forEach(info=>{
		var id = info[0];
		var origin = info[1];
		var x = origin===0 ? 0 : Math.floor(Graphics.width/2);
		var y = origin===0 ? 0 : Math.floor(Graphics.height/2);
		$gameScreen.showPicture(
			//pictureId,name,origin,x,y,scaleX,scaleY,opacity,blendMode
			id,"",origin,x,y,100,100,0,0
		);
	})
};

ProgressMovie.swapMoviePictures = function(){
	this._swapMoviePicture(PICTURE_IDS.lastMain,PICTURE_IDS.main);
	if(this.useBackMovie){
		this._swapMoviePicture(PICTURE_IDS.lastBack,PICTURE_IDS.back);
	}
};
ProgressMovie._swapMoviePicture = function(id0,id1){
	id0 = $gameScreen.realPictureId(id0);
	id1 = $gameScreen.realPictureId(id1);

	var picture0 = $gameScreen._pictures[id0];
	var picture1 = $gameScreen._pictures[id1];
	$gameScreen._pictures[id0] = picture1;
	$gameScreen._pictures[id1] = picture0;
};

ProgressMovie.fadePicture = function(id,duration,opacity=0){
	var picture = $gameScreen.picture(id);
	if(!picture)return;

	picture._duration = duration;
	picture._targetOpacity = opacity;
	if(duration<=0){
		picture._opacity = opacity;
	}

	if(isMZ){
		picture._wholeDuration = duration;
		picture._easingType = 0;
	}
};



/* blur
===================================*/
ProgressMovie.applyFilterBlurToPicture = function(id,blur){

	var container = SceneManager._scene._spriteset._pictureContainer;
	container.children.some(spr=>{
		if(!spr || spr._pictureId!==id)return false;

		var filter = new PIXI.filters.BlurFilter(blur);
		spr.filters = [filter];
		return true;
	})
};






})();