//=============================================================================
// TRP_Movies.js
//=============================================================================
/*:
 * @target MZ
 * @author Thirop
 * @plugindesc 動画ピクチャプラグイン
 * @base TRP_CORE
 * @orderAfter TRP_CORE
 * @help
 * 
 *
 *
 *
 * @command setup
 * @text [動画]セットアップ→再生
 * @desc 動画を(バックグラウンドで)再生開始
 * 
 * @arg id
 * @text 動画管理ID
 * @desc 他とかぶらない動画管理ID。
 *
 * @arg name
 * @text 動画ファイル名
 * @desc 動画ファイル名。省略または「def」でファイル名は「動画管理ID」のパラメータ
 * @default def
 *
 * @arg dir
 * @text 動画フォルダ
 * @desc 動画フォルダ。デフォルト時<def,d,省略>は「movies」
 * @default movies
 *
 * @arg resizeMode
 * @text リサイズモード
 * @desc 動画を自動でリサイズ
 * @default none
 * @type select
 * @option none/リサイズなし
 * @value none
 * @option fit/画面にフィット(余白あり)
 * @value fit
 * @option fitWidth/横幅をフィット
 * @value fitWidth
 * @option fitHeight/高さをフィット
 * @value fitHeight
 * @option fill/画面を埋める(はみ出しあり)
 * @value fill
 *
 * @arg fitRate
 * @text フィット率
 * @desc リサイズモードでfit,fitWidth,fitHeight指定時のフィット率(%)
 * @default 100
 * @type number
 * @min 1
 * @decimals 1
 * 
 * @arg loop
 * @text 動画をループ
 * @desc 動画をループ再生
 * @type boolean
 * @default false
 * 
 * @arg clearOnEnd
 * @text 再生完了時にクリア
 * @desc 再生完了時にクリア。ループ時は無効
 * @type boolean
 * @default true
 *
 * @arg lifespan
 * @text ライフスパン
 * @desc 動画を一時停止・クリアするタイミング
 * @type select
 * @default map
 * @option map/同じマップ中のみ有効
 * @value map
 * @option scene/シーン変更時にクリア
 * @value scene
 * @option manual/マニュアルでのみクリア
 * @value manual
 *
 *
 *
 * @command clear
 * @text [動画]クリア
 * @desc 再生中の動画を中断してクリア
 * 
 * @arg id
 * @text 対象の動画管理ID
 * @desc 対象の動画管理ID。(再生中の動画管理IDが１つの場合は省略可能)
 *
 *
 * @command pause
 * @text [動画]一時停止
 * @desc 再生中の動画を一時停止
 * 
 * @arg id
 * @text 対象の動画管理ID
 * @desc 対象の動画管理ID。(再生中の動画管理IDが１つの場合は省略可能)
 *
 *
 * @command resume
 * @text [動画]再生再開
 * @desc 一時停止中の動画の再生再開
 * 
 * @arg id
 * @text 対象の動画管理ID
 * @desc 対象の動画管理ID。(再生中の動画管理IDが１つの場合は省略可能)
 *
 * 
 * @command speed
 * @text [動画]再生速度の変更
 * @desc 再生速度の変更
 * 
 * @arg id
 * @text 対象の動画管理ID
 * @desc 対象の動画管理ID。(再生中の動画管理IDが１つの場合は省略可能)
 *
 * @arg speedRate
 * @type number
 * @default 1.0
 * @decimals 2
 * @min 0
 *
 * 
 * @command time
 * @text [動画]再生時間の変更
 * @desc 対象の動画管理IDの再生時間(秒数)を変更
 *
 * @arg id
 * @text 対象の動画管理ID
 * @desc 対象の動画管理ID。(再生中の動画管理IDが１つの場合は省略可能)
 * 
 * @arg time
 * @text 変更後の時間(秒)
 * @desc 変更後の時間(秒)
 * @type number
 * @default 0
 * @min 0
 * @decimals 2
 *
 * 
 * 
 * @command picture
 * @text [ピクチャ]動画をリンク
 * @desc ピクチャで動画を表示/クリア
 * 
 * @arg pictureId
 * @text ピクチャID
 * @desc 動画を表示するピクチャID
 * @type number
 * @default 1
 *
 * @arg id
 * @text 動画管理ID
 * @desc 表示する動画の管理ID。事前に「play/動画を再生」コマンドで再生&ピクチャ表示しておくこと。clearで解除
 *
 *
 *
 * ================================================
 *
 * @param priorMpegForDev
 * @text 開発中はmp4を優先
 * @desc 開発中はmp4を優先
 * @default false
 * @type boolean
 *
 * @param command
 * @text コマンド名(MV)
 * @desc プラグインコマンドのコマンド名。デフォは「movie」
 * @default movie
 * @type string
 *
 * @param hideOverflow
 * @text スクロールバーを隠す
 * @desc ウィンドウのスクロールバーを隠す（大きな動画表示時にスクロールバーが表示されるのを防止）
 * @default true
 * @type boorean
 *
 *
 * @param enableSave
 * @text セーブ/ロード対応(β)
 * @desc ゲーム本編用にセーブ/ロード対応。(試験導入機能のため本実装前に要テスト)
 * @default false
 * @type boolean
 *
 *
 *
 *
 */
//============================================================================= 


// PluginCommand
// MovieManager



function TRP_MovieManager(){}


(function(){
'use strict';


var isMZ = Utils.RPGMAKER_NAME==="MZ";

var pluginName = 'TRP_Movies';
var parameters = PluginManager.parameters(pluginName);
var hideOverflow = parameters.hideOverflow==='true';
var priorMpegForDev = parameters.priorMpegForDev==='true';
var enableSave = parameters.enableSave==='true'||parameters.enableSave===true;

var MovieManager = TRP_MovieManager;



//=============================================================================
// PluginCommand
//=============================================================================
var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
Game_Interpreter.prototype.pluginCommand = function(command,args){
	if(command === parameters.command){
		MovieManager.pluginCommand(args,this);
	}else{
		_Game_Interpreter_pluginCommand.call(this,...arguments);
	}
};

if(isMZ){
	['setup','claer','pause','resume','speed','time','picture'].forEach(command=>{
		PluginManager.registerCommand(pluginName,command,function(args){
			var argsArr = Object.values(args)
			argsArr.unshift(command);
			MovieManager.pluginCommand(argsArr,this);
		});
	});
};

MovieManager.pluginCommand = function(args,interpreter){
	var name = args[0];
	var command = "processCommand"+name[0].toUpperCase()+name.substring(1)
	this[command](args,interpreter);
};



//=============================================================================
// MovieManager
//=============================================================================
MovieManager.movies = {};
MovieManager.ids = [];
MovieManager.tryCreate = function(id){
	var movie = this.movies[id];
	if(!movie){
		movie = new Movie();
		this.movies[id] = movie;
		this.ids.push(id);
	}
	return movie;
};


MovieManager.processCommandSetup = function(args,interpreter){
	var i = 1;
	var id = args[i++];
	var name = args[i++];
	if(!name || name==='def'){
		name = id;
	}
	var dir = args[i++] || 'movies';
	if(dir==='def'||dir==='d')dir = 'movies';

	var resizeMode = args[i++];
	var fitRate = (Number(args[i++])||100)/100;
	var loop = args[i]===true||args[i++]==='true';
	var clearOnEnd = args[i]===true||args[i++]==='true';
	var lifespan = args[i++]||'map';

	this.setup(id,name,dir,resizeMode,fitRate,loop,clearOnEnd,lifespan);
};
MovieManager.setup = function(id,name,dir='movies',resizeMode='none',fitRate=1,loop=false,clearOnEnd=true,lifespan='map'){
	var movie = this.tryCreate(id);
	if(dir && dir[dir.length-1]!=='/'){
		dir = dir+'/';
	}
	var src = dir+name;
	if(Utils.isOptionValid('test') && priorMpegForDev){
		src += '.mp4';
	}else if(Utils.canPlayWebm()) {
        src +=  '.webm';
    } else {
        src += '.mp4';
    }
	movie.setup(src,resizeMode,fitRate,loop,clearOnEnd,lifespan);
};


MovieManager.processCommandClear = function(args,interpreter){
	var id = args[1];
	this.clear(id);
};
MovieManager.clear = function(id){
	id = id||this.ids[0];

	var movie = this.movies[id];
	if(!movie)return;

	movie.terminate();
	this.movies[id] = null;

	var idx = this.ids.indexOf(id);
	if(idx>=0){
		this.ids.splice(idx,1);
	}
}


MovieManager.processCommandPause = function(args,interpreter){
	var id = args[1]||this.ids[0];
	var movie = this.movies[id];
	if(!movie)return;
	movie.pause();
};
MovieManager.processCommandResume = function(args,interpreter){
	var id = args[1]||this.ids[0];
	var movie = this.movies[id];
	if(!movie)return;
	movie.resume();
};

MovieManager.processCommandSpeed = function(args,interpreter){
	var id = args[1]||this.ids[0];
	var rate = Number(args[2]);

	var movie = this.movies[id];
	if(!movie)return;
	if(args[2]===undefined||args[2]===''||isNaN(args[2]))return;

	movie.setSpeedRate(rate);
};
MovieManager.processCommandTime = function(args,interpreter){
	var id = args[1]||this.ids[0];
	var time = Number(args[2]);

	var movie = this.movies[id];
	if(!movie)return;
	if(args[2]===undefined||args[2]===''||isNaN(args[2]))return;

	movie.setTime(time);
};



/* update
===================================*/
MovieManager.update = function(){
	for(var i=this.ids.length-1; i>=0; i=(i-1)|0){
		var id = this.ids[i];
		var movie = this.movies[id];
		movie.update();

		if(!movie.playing && movie.clearOnEnd){
			movie.terminate();
			delete this.movies[id];
			this.ids.splice(i,1);
		}
	}
};


var _Scene_Base_update = Scene_Base.prototype.update;
Scene_Base.prototype.update = function(){
	_Scene_Base_update.call(this,...arguments);
	MovieManager.update();
}




//=============================================================================
// Movie
//=============================================================================
var Movie = MovieManager.Movie = function(){
	this.initialize.apply(this,arguments);
}
Movie.prototype.initialize = function(){
	this.playing = true;
	this._pause = false;

	this.bitmap = null;
	this.width = 0;
	this.height = 0;
	this._time = -1;
	this.loop = false;
	this.lifespan = 0;

	this._src = null;
	this._video = null;
	this._resizeMode = null;
	this._fitRate = 1;
	this._speedRate = 1;

	this._loading = false;
	this._loadListeners = [];
};
Movie.prototype.terminate = function(){
	if(this.bitmap){
		this.bitmap.clear();
	}
	this.bitmap = null;

	var video = this._video;
	if(video){
		video.pause();
		video.remove();
		video.onloadeddata = null;
		video.onended = null;
		video.onerror = null;
	}
	this._video = null;
	this._loadListeners.length = 0;
};

Movie.prototype.saveData = function(){
	var data = {};
	data.src = this._src;
	data.playing = this.playing;
	data.pause = this._pause;
	data.time = this._video ? this._video.currentTime : -1;
	data.resizeMode = this._resizeMode;
	data.fitRate = this._fitRate;
	data.speedRate = this._speedRate;
	data.loop = this.loop;
	data.clearOnEnd = this.clearOnEnd;
	data.lifespane = this.lifespan;

	return data;
};
Movie.prototype.loadData = function(data){
	this.playing = data.playing;
	this._pause = data.pause;
	this._time = data.time;
	this._resizeMode = data.resizeMode;
	this._fitRate = data.fitRate;
	this._speedRate = data.speedRate;
	this.loop = data.loop;
	this.clearOnEnd = data.clearOnEnd;
	this.lifespane = data.lifespan;

	this.setup(data.src,this._resizeMode,this._fitRate,this.loop,this.clearOnEnd,this._time)
};

Movie.prototype.setup = function(src,resizeMode='none',fitRate=1,loop=false,clearOnEnd=false,lifespan='map',time=-1){
	if(this._src===src)return;
	this._src = src;
	this._resizeMode = resizeMode||null;
	this._fitRate = fitRate||1;
	this._time = time;
	this.loop = loop;
	this.clearOnEnd = clearOnEnd;
	this.lifespan = LIFESPAN[lifespan]||0;

	var video = this._video;
	if(!video){
		var video = document.createElement('video')
		this._video = video;
		video.style.opacity = 0;

		video.setAttribute('playsinline','');
		if(hideOverflow){
			document.body.style.overflow = "hidden";
		}
	    video.style.overflow = "hidden";
	    if(!isMZ){
	    	makeVideoPlayableInline(video);
	    }
	    document.body.appendChild(video);
	}

	this._loading = true;
	video.src = this._src;
	video.onloadeddata = this.onVideoLoad.bind(this,video);
};

Movie.prototype.onVideoLoad = function(video){
	if(video!==this._video)return;
	video.onloadeddata = null;

	//adjustSize
	var w = video.clientWidth||1;
	var h = video.clientHeight||1;
	var wRate = Graphics.width/w;
	var hRate = Graphics.height/h;
	var rate = 1;
	if(this._resizeMode==='fill'){
		rate = Math.max(wRate,hRate);
	}else if(this._resizeMode==='fit'){
		rate = Math.min(wRate,hRate)*this._fitRate;
	}else if(this._resizeMode==='fitWidth'){
		rate = wRate*this._fitRate;
	}else if(this._resizeMode==='fitHeight'){
		rate = hRate*this._fitRate;
	}
	rate = rate.clamp(0,1);
	this.width = Math.floor(w*rate);
	this.height = Math.floor(h*rate);


	//prepare bitmap
	if(this.bitmap){
		if(this.bitmap.width<=this.width && this.bitmap.height<=this.height){
			this.bitmap._canvas.width = this.width;
			this.bitmap._canvas.height = this.height;
			this.bitmap._baseTexture.setSize(this.width,this.height);
		}else{
			this.bitmap = null;
		}
	}
	if(!this.bitmap){
		this.bitmap = new Bitmap(this.width,this.height);
	}


	if(this._time>=0){
		video.currentTime = this._time;
		this._time = -1;
	}

	if(this.playing && !this._pause){
		this.play();
	}


	this._loading = false;
	for(const listener of this._loadListeners){
		listener();
	}
	this._loadListeners.length = 0;
};
Movie.prototype.addLoadListener = function(listener){
	if(!this._loading){
		listener();
	}else{
		this._loadListeners.push(listener);
	}
}

Movie.prototype.play = function(){
	this._pause = false;

	var video = this._video;
	video.playbackRate = this._speedRate;
	video.play();
	video.onended = this.onVideoEnd.bind(this);
	video.onerror = this._onVideoEnd.bind(this);
};

Movie.prototype.onVideoEnd = function(){
	if(this.loop){
		this._video.currentTime = 0;
		if(this.playing){
			this._video.play();
		}
	}else{
		this.playing = false;
	}
};

Movie.prototype._onVideoEnd = function(){
	this.playing = false;
};

Movie.prototype.setSpeedRate = function(rate=1){
	if(this._speedRate===rate)return;
	this._speedRate = rate;

	if(this._video){
		this._video.playbackRate = rate;
	}
};
Movie.prototype.setTime = function(time=0){
	if(this._video){
		this._video.currentTime = time;
		this._time = -1;
	}else{
		this._time = time;
	}
};
Movie.prototype.pause = function(){
	if(this._pause)return;
	this._pause = true;
	if(this._video){
		this._video.pause();
	}
};
Movie.prototype.resume = function(){
	if(!this._pause)return;
	this._pause = false;

	if(this._video){
		this.play();
	}
}


/* update
===================================*/
Movie.prototype.update = function(){
	if(!this.playing)return;

	if(this.bitmap && this._video){
		this.bitmap._context.drawImage(this._video,0,0,this.width,this.height);
		if(isMZ){
			this.bitmap._baseTexture.update();
		}else{
			this.bitmap._setDirty();
		}
	}

};


MovieManager.processCommandPicture = function(args,interpreter){
	var pictureId = Number(args[1]);
	var picture = $gameScreen.picture(pictureId);
	if(!picture)return;

	var id = args[2];
	if(id==='clear'){
		picture._trpMovieId = null;
	}else{
		if(!MovieManager.movies[id])return;
		picture._trpMovieId = id;
	}
};



/* lifespan
===================================*/
var LIFESPAN = Movie.LIFESPAN = {
	map:0,
	scene:1,
	manual:2,
};


var _SceneManager_changeScene = SceneManager.changeScene;
SceneManager.changeScene = function(){
	var lastScene = this._scene;
	_SceneManager_changeScene.call(this,...arguments);

	if(this._scene!==lastScene){
		MovieManager.onSceneChange(this._scene,lastScene);
	}
};

var _Scene_Load_onLoadSuccess = Scene_Load.prototype.onLoadSuccess;
Scene_Load.prototype.onLoadSuccess = function(){
	MovieManager.operateMovieWithLifespan(LIFESPAN.map,'clear');
	if(enableSave){
		MovieManager.applyLoadContents();
	}
	_Scene_Load_onLoadSuccess.call(this,...arguments);
};

MovieManager.onSceneChange = function(scene,lastScene){
	this.operateMovieWithLifespan(LIFESPAN.sceneKill,'clear');

	if(($gamePlayer && $gamePlayer.isTransferring())
		|| (scene instanceof Scene_Title)
	){
		this.operateMovieWithLifespan(LIFESPAN.map,'clear');
	}else{
		if(scene instanceof Scene_Map){
			this.operateMovieWithLifespan(LIFESPAN.map,'resume');
		}else{
			this.operateMovieWithLifespan(LIFESPAN.map,'pause');
		}
	}
};

MovieManager.operateMovieWithLifespan = function(lifespan,operate='clear'){
	var ids = this.ids;
	for(var i=ids.length-1; i>=0; i=(i-1)|0){
		var id = ids[i];
		var movie = this.movies[id];
		if(movie.lifespan === lifespan){
			if(operate==='pause'){
				movie.pause();
			}else if(operate==='resume'){
				movie.resume();
			}else{
				this.clear(id);
			}
		}
	}
};





//=============================================================================
// Game_Picture
//=============================================================================
var _Game_Picture_initialize = Game_Picture.prototype.initialize;
Game_Picture.prototype.initialize = function(){
	_Game_Picture_initialize.call(this,...arguments);
	this._trpMovieId = null;
};

var _Game_Picture_name = Game_Picture.prototype.name;
Game_Picture.prototype.name = function(){
	if(this._trpMovieId){
		return picturePrefix+this._trpMovieId;
	}else{
		return _Game_Picture_name.call(this,...arguments);
	}
}





//=============================================================================
// Sprite_Picture
//=============================================================================
var picturePrefix = 'TRP_MOVIE:';

var _Sprite_Picture_initialize = Sprite_Picture.prototype.initialize;
Sprite_Picture.prototype.initialize = function(pictureId) {
	_Sprite_Picture_initialize.call(this,...arguments);
	this._trpMovieId = this._trpMovieId||null;
};

var _Sprite_Picture_loadBitmap = Sprite_Picture.prototype.loadBitmap;
Sprite_Picture.prototype.loadBitmap = function() {
	this._trpMovieId = null;
	if(this._pictureName.indexOf(picturePrefix)===0){
		this._trpMovieId = this._pictureName.replace(picturePrefix,'');
		this.updateTrpMovieBitmap();
	}else{
		_Sprite_Picture_loadBitmap.call(this,...arguments);
	}
};

var _Sprite_Picture_update = Sprite_Picture.prototype.update;
Sprite_Picture.prototype.update = function(){
	_Sprite_Picture_update.call(this);

	if(this._trpMovieId){
		this.updateTrpMovieBitmap();
	}
};

Sprite_Picture.prototype.updateTrpMovieBitmap = function(){
	var video = MovieManager.movies[this._trpMovieId];
	if(video){
		if(this.bitmap !== video.bitmap){
			this.bitmap = video.bitmap;
		}
	}else{
		this.bitmap = null;
	}
};








//=============================================================================
// Save Load
//=============================================================================
if(enableSave)(()=>{
TRP_MovieManager.saveContents = function(contents){
	if(!this.ids || !this.ids.length){
		contents.trpMovies = null;
		return;
	}

	var saveData = {};
	contents.trpMovies = saveData;

	for(const id of this.ids){
		saveData[id] = this.movies[id].saveData();
	}
};

TRP_MovieManager._reservedLoadContents = null;
TRP_MovieManager.loadContents = function(contents){
	TRP_MovieManager._reservedLoadContents = contents.trpMovies||null;
};

TRP_MovieManager.applyLoadContents = function(){
	var saveData = this._reservedLoadContents;
	this._reservedLoadContents = null;
	if(!saveData)return;

	var ids = Object.keys(saveData);
	for(const id of ids){
		var movie = this.tryCreate(id);
		movie.loadData(saveData[id]);
	}
};
TRP_MovieManager.onSetupNewGame = function(){
	this._reservedLoadContents = null;
};


var _DataManager_makeSaveContents = DataManager.makeSaveContents;
DataManager.makeSaveContents = function() {
	var contents = _DataManager_makeSaveContents.call(this,...arguments);
	TRP_MovieManager.saveContents(contents);
	return contents;
};

var _DataManager_extractSaveContents = DataManager.extractSaveContents;
DataManager.extractSaveContents = function(contents){
	_DataManager_extractSaveContents.call(this,...arguments);
	TRP_MovieManager.loadContents(contents);
};

var _DataManager_setupNewGame = DataManager.setupNewGame;
DataManager.setupNewGame = function(){
	_DataManager_setupNewGame.call(this,...arguments);
	TRP_MovieManager.onSetupNewGame();
};


})()










})();


